<?php

namespace App\Console\Commands;

use App\Models\Schedule;
use App\Models\UploadLog;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class ResumeSchedulesCommand extends Command
{
    protected $signature = 'updit:resume-schedules';
    protected $description = 'Resume schedules that were paused due to quota limits';

    public function handle(): int
    {
        $this->info('Checking for schedules to resume...');

        // Find schedules that are ready to resume
        $schedules = Schedule::where('status', 'quota_limit')
            ->where('auto_restart', true)
            ->whereNotNull('resume_at')
            ->where('resume_at', '<=', now())
            ->get();

        if ($schedules->isEmpty()) {
            $this->info('No schedules to resume.');
            return self::SUCCESS;
        }

        $this->info("Found {$schedules->count()} schedule(s) to resume.");

        foreach ($schedules as $schedule) {
            $this->resumeSchedule($schedule);
        }

        return self::SUCCESS;
    }

    protected function resumeSchedule(Schedule $schedule): void
    {
        $this->line("Resuming schedule #{$schedule->id}...");

        // Check if there are still pending videos
        if ($schedule->pendingVideos()->count() === 0) {
            $this->info("  No pending videos. Marking as completed.");
            $schedule->update(['status' => 'completed', 'resume_at' => null]);
            return;
        }

        // Resume the schedule
        $schedule->resumeFromQuota();

        // Log the resume
        UploadLog::create([
            'user_id' => $schedule->user_id,
            'schedule_id' => $schedule->id,
            'status' => 'success',
            'message' => 'Schedule auto-resumed after 24h quota reset',
        ]);

        $this->info("  Schedule resumed successfully.");
        Log::info("Schedule {$schedule->id} auto-resumed after quota limit period");
    }
}
