<?php

namespace App\Console\Commands;

use App\Jobs\ProcessVideoUpload;
use App\Models\Schedule;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class RunScheduleCommand extends Command
{
    protected $signature = 'updit:run-schedule';
    protected $description = 'Process scheduled video uploads';

    public function handle(): int
    {
        $this->info('Checking for scheduled uploads...');

        // Find schedules that are ready to run
        $schedules = Schedule::where('status', 'active')
            ->whereNotNull('next_run_at')
            ->where('next_run_at', '<=', now())
            ->with(['socialAccount', 'user.plan'])
            ->get();

        if ($schedules->isEmpty()) {
            $this->info('No schedules ready to run.');
            return self::SUCCESS;
        }

        $this->info("Found {$schedules->count()} schedule(s) ready to process.");

        foreach ($schedules as $schedule) {
            $this->processSchedule($schedule);
        }

        return self::SUCCESS;
    }

    protected function processSchedule(Schedule $schedule): void
    {
        $this->line("Processing schedule #{$schedule->id}...");

        // Check if user is active
        if (!$schedule->user->is_active) {
            $this->warn("  User is inactive, pausing schedule.");
            $schedule->update(['status' => 'paused']);
            return;
        }

        // Check if social account needs reauth
        if ($schedule->socialAccount->needs_reauth) {
            $this->warn("  Social account needs re-authentication, pausing schedule.");
            $schedule->update(['status' => 'paused']);
            return;
        }

        // Get next video to upload
        $video = $schedule->nextVideo;

        if (!$video) {
            $this->info("  No more videos to upload. Marking as completed.");
            $schedule->update(['status' => 'completed', 'next_run_at' => null]);
            return;
        }

        // Check daily limit
        $user = $schedule->user;
        if ($user->hasReachedDailyLimit()) {
            $this->warn("  User has reached daily limit. Pausing for quota.");
            $schedule->pauseForQuota();
            return;
        }

        // Dispatch the upload job
        $this->info("  Dispatching upload job for video #{$video->id}: {$video->title}");
        
        ProcessVideoUpload::dispatch($video, $schedule);

        // Set next run time (the job will update this on completion)
        $schedule->update([
            'next_run_at' => now()->addMinutes($schedule->interval_minutes),
        ]);

        Log::info("Dispatched upload job for video {$video->id} on schedule {$schedule->id}");
    }
}
