<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Plan;
use App\Models\Schedule;
use App\Models\SocialAccount;
use App\Models\User;
use App\Models\Video;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class AdminUserController extends Controller
{
    /**
     * Display list of all users
     */
    public function index()
    {
        $users = User::with('plan')
            ->withCount(['schedules', 'videos', 'socialAccounts'])
            ->latest()
            ->paginate(20);

        return view('admin.users.index', compact('users'));
    }

    /**
     * Show user details/edit form
     */
    public function show(User $user)
    {
        $plans = Plan::all();
        $user->load(['schedules.videos', 'socialAccounts', 'uploadLogs' => function($q) {
            $q->latest()->limit(10);
        }]);

        $stats = [
            'total_videos' => $user->videos()->count(),
            'uploaded_videos' => $user->videos()->where('is_uploaded', true)->count(),
            'pending_videos' => $user->videos()->where('is_uploaded', false)->count(),
            'total_schedules' => $user->schedules()->count(),
            'active_schedules' => $user->schedules()->where('status', 'active')->count(),
            'connected_accounts' => $user->socialAccounts()->count(),
            'storage_used' => $this->calculateStorageUsed($user),
        ];

        return view('admin.users.show', compact('user', 'plans', 'stats'));
    }

    /**
     * Show user edit form
     */
    public function edit(User $user)
    {
        $plans = Plan::all();
        $user->load(['schedules', 'socialAccounts']);

        return view('admin.users.edit', compact('user', 'plans'));
    }

    /**
     * Update user info
     */
    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'plan_id' => 'required|exists:plans,id',
            'role' => 'required|in:user,admin',
            'is_active' => 'boolean',
        ]);

        $user->update([
            'name' => $request->name,
            'email' => $request->email,
            'plan_id' => $request->plan_id,
            'role' => $request->role,
            'is_active' => $request->boolean('is_active'),
        ]);

        return redirect()->route('admin.users.index')
            ->with('success', 'User updated successfully.');
    }

    /**
     * Reset user password manually
     */
    public function resetPassword(Request $request, User $user)
    {
        $request->validate([
            'new_password' => 'required|min:6|confirmed',
        ]);

        $user->update([
            'password' => Hash::make($request->new_password),
        ]);

        return back()->with('success', "Password reset for {$user->name}.");
    }

    /**
     * Generate and set a random password
     */
    public function generatePassword(User $user)
    {
        $newPassword = Str::random(12);
        
        $user->update([
            'password' => Hash::make($newPassword),
        ]);

        return back()->with('success', "New password generated for {$user->name}: {$newPassword}");
    }

    /**
     * Update user email
     */
    public function updateEmail(Request $request, User $user)
    {
        $request->validate([
            'email' => 'required|email|unique:users,email,' . $user->id,
        ]);

        $oldEmail = $user->email;
        $user->update(['email' => $request->email]);

        return back()->with('success', "Email changed from {$oldEmail} to {$request->email}.");
    }

    /**
     * Change user plan/subscription
     */
    public function changePlan(Request $request, User $user)
    {
        $request->validate([
            'plan_id' => 'required|exists:plans,id',
        ]);

        $oldPlan = $user->plan->name;
        $user->update(['plan_id' => $request->plan_id]);
        $newPlan = Plan::find($request->plan_id)->name;

        return back()->with('success', "Plan changed from {$oldPlan} to {$newPlan}.");
    }

    /**
     * Reset daily upload quota
     */
    public function resetQuota(User $user)
    {
        $user->schedules()->update(['daily_uploaded_count' => 0]);

        return back()->with('success', "Quota reset for {$user->name}.");
    }

    /**
     * Suspend user (ban)
     */
    public function ban(User $user)
    {
        if ($user->role === 'admin') {
            return back()->with('error', 'Cannot ban admin users.');
        }

        // Pause all schedules
        $user->schedules()->update(['status' => 'paused']);
        $user->update(['is_active' => false]);

        return back()->with('success', "User {$user->name} has been suspended.");
    }

    /**
     * Unsuspend user (unban)
     */
    public function unban(User $user)
    {
        $user->update(['is_active' => true]);

        return back()->with('success', "User {$user->name} has been activated.");
    }

    /**
     * Verify user email manually
     */
    public function verifyEmail(User $user)
    {
        $user->update(['email_verified_at' => now()]);

        return back()->with('success', "Email verified for {$user->name}.");
    }

    /**
     * Unverify user email
     */
    public function unverifyEmail(User $user)
    {
        $user->update(['email_verified_at' => null]);

        return back()->with('success', "Email unverified for {$user->name}.");
    }

    /**
     * Impersonate user (login as user)
     */
    public function impersonate(User $user)
    {
        if ($user->role === 'admin') {
            return back()->with('error', 'Cannot impersonate admin users.');
        }

        // Store original admin ID in session
        session(['impersonate_admin_id' => auth()->id()]);
        
        // Login as the user
        auth()->login($user);

        return redirect()->route('dashboard')
            ->with('info', "You are now logged in as {$user->name}. Click 'Stop Impersonating' to return.");
    }

    /**
     * Stop impersonating and return to admin
     */
    public function stopImpersonating()
    {
        $adminId = session('impersonate_admin_id');
        
        if ($adminId) {
            $admin = User::find($adminId);
            if ($admin && $admin->role === 'admin') {
                auth()->login($admin);
                session()->forget('impersonate_admin_id');
                return redirect()->route('admin.users.index')
                    ->with('success', 'Returned to admin account.');
            }
        }

        return redirect()->route('dashboard');
    }

    /**
     * Delete all user's schedules and videos
     */
    public function clearData(User $user)
    {
        // Delete all video files
        foreach ($user->videos as $video) {
            if ($video->file_path) {
                Storage::disk('local')->delete('private/' . $video->file_path);
            }
        }

        // Delete schedules (videos will cascade)
        $user->schedules()->delete();

        return back()->with('success', "All data cleared for {$user->name}.");
    }

    /**
     * Disconnect all social accounts
     */
    public function disconnectAccounts(User $user)
    {
        $count = $user->socialAccounts()->count();
        $user->socialAccounts()->delete();

        return back()->with('success', "Disconnected {$count} account(s) for {$user->name}.");
    }

    /**
     * Permanently delete user
     */
    public function destroy(User $user)
    {
        if ($user->role === 'admin') {
            return back()->with('error', 'Cannot delete admin users.');
        }

        if ($user->id === auth()->id()) {
            return back()->with('error', 'Cannot delete yourself.');
        }

        // Delete all video files
        foreach ($user->videos as $video) {
            if ($video->file_path) {
                Storage::disk('local')->delete('private/' . $video->file_path);
            }
        }

        $userName = $user->name;
        $user->delete();

        return redirect()->route('admin.users.index')
            ->with('success', "User {$userName} has been permanently deleted.");
    }

    /**
     * Calculate storage used by user
     */
    private function calculateStorageUsed(User $user): string
    {
        $bytes = $user->videos()->sum('file_size');
        
        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        }
        
        return $bytes . ' bytes';
    }

    /**
     * Export user data
     */
    public function exportData(User $user)
    {
        $data = [
            'user' => $user->only(['id', 'name', 'email', 'role', 'created_at']),
            'plan' => $user->plan->name,
            'schedules' => $user->schedules->map(fn($s) => $s->only(['id', 'name', 'status', 'created_at'])),
            'videos_count' => $user->videos()->count(),
            'social_accounts' => $user->socialAccounts->map(fn($a) => [
                'platform' => $a->platform,
                'username' => $a->platform_user_name,
            ]),
        ];

        return response()->json($data, 200, [], JSON_PRETTY_PRINT);
    }
}
