<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Artisan;

class ApiKeyController extends Controller
{
    /**
     * Show API keys management page
     */
    public function index()
    {
        return view('admin.api-keys');
    }

    /**
     * Update social platform API keys
     * NOTE: This writes to .env file - use with caution
     */
    public function updateSocialKeys(Request $request)
    {
        // Validate admin is authenticated (already done by middleware)
        
        $envPath = base_path('.env');
        
        if (!File::exists($envPath)) {
            return back()->with('error', '.env file not found');
        }

        $envContent = File::get($envPath);
        $updated = false;

        // Google/YouTube
        if ($request->filled('google_client_id') && $request->google_client_id !== '••••••••••••••••') {
            $this->setEnvValue($envContent, 'GOOGLE_CLIENT_ID', $request->google_client_id);
            $updated = true;
        }
        if ($request->filled('google_client_secret')) {
            $this->setEnvValue($envContent, 'GOOGLE_CLIENT_SECRET', $request->google_client_secret);
            $updated = true;
        }

        // Facebook
        if ($request->filled('facebook_client_id') && $request->facebook_client_id !== '••••••••••••••••') {
            $this->setEnvValue($envContent, 'FACEBOOK_CLIENT_ID', $request->facebook_client_id);
            $updated = true;
        }
        if ($request->filled('facebook_client_secret')) {
            $this->setEnvValue($envContent, 'FACEBOOK_CLIENT_SECRET', $request->facebook_client_secret);
            $updated = true;
        }

        // TikTok
        if ($request->filled('tiktok_client_key') && $request->tiktok_client_key !== '••••••••••••••••') {
            $this->setEnvValue($envContent, 'TIKTOK_CLIENT_KEY', $request->tiktok_client_key);
            $updated = true;
        }
        if ($request->filled('tiktok_client_secret')) {
            $this->setEnvValue($envContent, 'TIKTOK_CLIENT_SECRET', $request->tiktok_client_secret);
            $updated = true;
        }

        if ($updated) {
            // Write to .env file
            File::put($envPath, $envContent);
            
            // Clear config cache
            Artisan::call('config:clear');
            
            // Log the update (without revealing keys)
            Log::info('API keys updated by admin', [
                'user_id' => auth()->id(),
                'ip' => request()->ip(),
            ]);

            return back()->with('success', 'API keys updated successfully. Config cache cleared.');
        }

        return back()->with('info', 'No changes were made.');
    }

    /**
     * Set or update an environment variable in the .env content
     */
    protected function setEnvValue(string &$envContent, string $key, string $value): void
    {
        // Escape special characters for .env file
        $value = $this->sanitizeEnvValue($value);
        
        // Check if key exists
        $pattern = "/^{$key}=.*/m";
        
        if (preg_match($pattern, $envContent)) {
            // Update existing value
            $envContent = preg_replace($pattern, "{$key}={$value}", $envContent);
        } else {
            // Add new key
            $envContent .= "\n{$key}={$value}";
        }
    }

    /**
     * Sanitize value for .env file
     */
    protected function sanitizeEnvValue(string $value): string
    {
        // If value contains spaces or special characters, wrap in quotes
        if (preg_match('/[\s#]/', $value)) {
            $value = '"' . str_replace('"', '\"', $value) . '"';
        }
        return $value;
    }
}
