<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Schedule;
use App\Models\Setting;
use App\Models\User;
use App\Models\Video;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;

class BatchController extends Controller
{
    /**
     * Show all active upload batches
     */
    public function index()
    {
        $activeSchedules = Schedule::with(['user', 'socialAccount', 'videos'])
            ->where('status', 'active')
            ->whereHas('videos', function ($q) {
                $q->where('is_uploaded', false);
            })
            ->get();

        $cronLastRun = Setting::get('cron_last_run');
        $cronRunning = $cronLastRun && now()->diffInMinutes($cronLastRun) < 10;

        $stats = [
            'active_batches' => $activeSchedules->count(),
            'pending_videos' => Video::where('is_uploaded', false)->count(),
            'uploading_now' => Video::where('status', 'uploading')->count(),
        ];

        return view('admin.batches.index', compact('activeSchedules', 'cronLastRun', 'cronRunning', 'stats'));
    }

    /**
     * Stop a specific schedule/batch
     */
    public function stop(Schedule $schedule)
    {
        $schedule->pause();

        return back()->with('success', "Batch '{$schedule->name}' has been stopped.");
    }

    /**
     * Stop all active batches
     */
    public function stopAll()
    {
        Schedule::where('status', 'active')->update(['status' => 'paused']);

        return back()->with('success', 'All active batches have been stopped.');
    }

    /**
     * Resume a schedule
     */
    public function resume(Schedule $schedule)
    {
        $schedule->resume();

        return back()->with('success', "Batch '{$schedule->name}' has been resumed.");
    }

    /**
     * Get cron job status
     */
    public function cronStatus()
    {
        $cronLastRun = Setting::get('cron_last_run');
        $cronRunning = $cronLastRun && now()->diffInMinutes($cronLastRun) < 10;

        return response()->json([
            'running' => $cronRunning,
            'last_run' => $cronLastRun ? \Carbon\Carbon::parse($cronLastRun)->toIso8601String() : null,
            'minutes_ago' => $cronLastRun ? now()->diffInMinutes($cronLastRun) : null,
        ]);
    }

    /**
     * Test cron manually
     */
    public function testCron()
    {
        try {
            Artisan::call('schedule:run');
            Setting::set('cron_last_run', now()->toDateTimeString());
            
            return back()->with('success', 'Cron job executed successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Cron job failed: ' . $e->getMessage());
        }
    }
}
