<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\PaymentGateway;
use App\Models\Plan;
use App\Models\Setting;
use App\Models\Subscription;
use App\Models\User;
use Illuminate\Http\Request;

class PaymentController extends Controller
{
    /**
     * Payment gateways settings
     */
    public function gateways()
    {
        $gateways = PaymentGateway::orderBy('sort_order')->get();
        
        return view('admin.payments.gateways', compact('gateways'));
    }

    /**
     * Update gateway settings
     */
    public function updateGateway(Request $request, PaymentGateway $gateway)
    {
        $request->validate([
            'display_name' => 'required|string|max:255',
            'is_active' => 'boolean',
            'instructions' => 'nullable|string',
            'sort_order' => 'integer',
        ]);

        $config = [];
        
        // Gateway-specific config
        switch ($gateway->name) {
            case 'paypal':
                $config = [
                    'client_id' => $request->client_id,
                    'client_secret' => $request->client_secret,
                    'mode' => $request->mode ?? 'sandbox',
                ];
                break;
            case 'paystack':
                $config = [
                    'public_key' => $request->public_key,
                    'secret_key' => $request->secret_key,
                ];
                break;
            case 'flutterwave':
                $config = [
                    'public_key' => $request->public_key,
                    'secret_key' => $request->secret_key,
                    'encryption_key' => $request->encryption_key,
                ];
                break;
            case 'skrill':
                $config = [
                    'merchant_id' => $request->merchant_id,
                    'secret_word' => $request->secret_word,
                ];
                break;
            case 'opay':
                $config = [
                    'merchant_id' => $request->merchant_id,
                    'public_key' => $request->public_key,
                    'secret_key' => $request->secret_key,
                ];
                break;
        }

        $gateway->update([
            'display_name' => $request->display_name,
            'is_active' => $request->boolean('is_active'),
            'config' => $config ?: $gateway->config,
            'instructions' => $request->instructions,
            'sort_order' => $request->sort_order ?? 0,
        ]);

        return back()->with('success', "{$gateway->display_name} settings updated.");
    }

    /**
     * List all payments
     */
    public function payments()
    {
        $payments = Payment::with(['user', 'plan'])
            ->latest()
            ->paginate(20);
        
        $stats = [
            'total' => Payment::where('status', 'completed')->sum('amount'),
            'pending' => Payment::where('status', 'pending')->count(),
            'this_month' => Payment::where('status', 'completed')
                ->whereMonth('paid_at', now()->month)
                ->sum('amount'),
        ];

        return view('admin.payments.index', compact('payments', 'stats'));
    }

    /**
     * Show payment details
     */
    public function show(Payment $payment)
    {
        $payment->load(['user', 'plan', 'subscription']);
        
        return view('admin.payments.show', compact('payment'));
    }

    /**
     * Approve manual payment
     */
    public function approve(Request $request, Payment $payment)
    {
        if ($payment->status !== 'pending') {
            return back()->with('error', 'This payment has already been processed.');
        }

        $payment->markAsCompleted('MANUAL-' . now()->timestamp);
        $payment->update(['admin_notes' => $request->admin_notes]);

        // Create or update subscription
        $subscription = Subscription::updateOrCreate(
            ['user_id' => $payment->user_id, 'status' => 'active'],
            [
                'plan_id' => $payment->plan_id,
                'status' => 'active',
                'payment_method' => 'manual',
                'amount' => $payment->amount,
                'starts_at' => now(),
                'ends_at' => now()->addMonth(),
            ]
        );

        $payment->update(['subscription_id' => $subscription->id]);

        // Update user plan
        $payment->user->update(['plan_id' => $payment->plan_id]);

        return back()->with('success', 'Payment approved and subscription activated.');
    }

    /**
     * Reject payment
     */
    public function reject(Request $request, Payment $payment)
    {
        if ($payment->status !== 'pending') {
            return back()->with('error', 'This payment has already been processed.');
        }

        $payment->update([
            'status' => 'failed',
            'admin_notes' => $request->admin_notes ?? 'Payment rejected by admin.',
        ]);

        return back()->with('success', 'Payment rejected.');
    }

    /**
     * List all subscriptions
     */
    public function subscriptions()
    {
        $subscriptions = Subscription::with(['user', 'plan'])
            ->latest()
            ->paginate(20);

        return view('admin.payments.subscriptions', compact('subscriptions'));
    }
}
