<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Plan;
use Illuminate\Http\Request;

class PlanController extends Controller
{
    /**
     * Display list of all plans
     */
    public function index()
    {
        $plans = Plan::withCount('users')->get();
        
        return view('admin.plans.index', compact('plans'));
    }

    /**
     * Show create plan form
     */
    public function create()
    {
        return view('admin.plans.create');
    }

    /**
     * Store new plan
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:plans,name',
            'max_videos_per_batch' => 'required|integer|min:1',
            'min_interval_minutes' => 'required|integer|min:1',
            'daily_upload_cap' => 'required|integer|min:1',
            'price' => 'required|numeric|min:0',
            'price_id' => 'nullable|string|max:255',
            'features' => 'nullable|array',
        ]);

        Plan::create([
            'name' => $request->name,
            'max_videos_per_batch' => $request->max_videos_per_batch,
            'min_interval_minutes' => $request->min_interval_minutes,
            'daily_upload_cap' => $request->daily_upload_cap,
            'price' => $request->price,
            'price_id' => $request->price_id,
            'features' => $request->features ?? [],
        ]);

        return redirect()->route('admin.plans.index')
            ->with('success', 'Plan created successfully.');
    }

    /**
     * Show edit plan form
     */
    public function edit(Plan $plan)
    {
        return view('admin.plans.edit', compact('plan'));
    }

    /**
     * Update plan
     */
    public function update(Request $request, Plan $plan)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:plans,name,' . $plan->id,
            'max_videos_per_batch' => 'required|integer|min:1',
            'min_interval_minutes' => 'required|integer|min:1',
            'daily_upload_cap' => 'required|integer|min:1',
            'price' => 'required|numeric|min:0',
            'price_id' => 'nullable|string|max:255',
            'features' => 'nullable|array',
        ]);

        $plan->update([
            'name' => $request->name,
            'max_videos_per_batch' => $request->max_videos_per_batch,
            'min_interval_minutes' => $request->min_interval_minutes,
            'daily_upload_cap' => $request->daily_upload_cap,
            'price' => $request->price,
            'price_id' => $request->price_id,
            'features' => $request->features ?? [],
        ]);

        return redirect()->route('admin.plans.index')
            ->with('success', 'Plan updated successfully.');
    }

    /**
     * Delete plan
     */
    public function destroy(Plan $plan)
    {
        // Check if users are on this plan
        if ($plan->users()->count() > 0) {
            return back()->with('error', 'Cannot delete plan with active users. Migrate users first.');
        }

        $plan->delete();

        return redirect()->route('admin.plans.index')
            ->with('success', 'Plan deleted successfully.');
    }

    /**
     * Set plan as default for new users
     */
    public function setDefault(Plan $plan)
    {
        // For now, just return success - in real implementation, 
        // you'd store this in settings
        return back()->with('success', "{$plan->name} set as default plan.");
    }
}
