<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\Plan;
use App\Models\Subscription;
use Carbon\Carbon;
use Illuminate\Http\Request;

class SubscriptionController extends Controller
{
    /**
     * Show subscriptions dashboard with revenue stats
     */
    public function index(Request $request)
    {
        // Build payments query with filters
        $query = Payment::with(['user', 'plan'])->latest();

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->month) {
            $date = Carbon::createFromFormat('Y-m', $request->month);
            $query->whereYear('created_at', $date->year)
                  ->whereMonth('created_at', $date->month);
        }

        $payments = $query->paginate(20);

        // Calculate stats
        $thisMonth = Payment::where('status', 'completed')
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->sum('amount');

        $lastMonth = Payment::where('status', 'completed')
            ->whereMonth('created_at', now()->subMonth()->month)
            ->whereYear('created_at', now()->subMonth()->year)
            ->sum('amount');

        $lastMonthCount = Payment::where('status', 'completed')
            ->whereMonth('created_at', now()->subMonth()->month)
            ->whereYear('created_at', now()->subMonth()->year)
            ->count();

        $growth = $lastMonth > 0 ? round((($thisMonth - $lastMonth) / $lastMonth) * 100, 1) : 100;

        $stats = [
            'this_month' => $thisMonth,
            'last_month' => $lastMonth,
            'last_month_count' => $lastMonthCount,
            'growth' => $growth,
            'total_revenue' => Payment::where('status', 'completed')->sum('amount'),
            'total_purchases' => Payment::where('status', 'completed')->count(),
            'active_subs' => Subscription::where('status', 'active')->count(),
            'pending_subs' => Payment::where('status', 'pending')->count(),
        ];

        // Chart data - last 6 months
        $chartData = [
            'labels' => [],
            'revenue' => [],
            'plan_labels' => [],
            'plan_revenue' => [],
        ];

        for ($i = 5; $i >= 0; $i--) {
            $month = now()->subMonths($i);
            $chartData['labels'][] = $month->format('M Y');
            $chartData['revenue'][] = Payment::where('status', 'completed')
                ->whereMonth('created_at', $month->month)
                ->whereYear('created_at', $month->year)
                ->sum('amount');
        }

        // Revenue by plan
        $plans = Plan::withCount(['users' => function($q) {
            $q->whereHas('payments', fn($p) => $p->where('status', 'completed'));
        }])->get();

        foreach ($plans as $plan) {
            $chartData['plan_labels'][] = $plan->name;
            $chartData['plan_revenue'][] = Payment::where('status', 'completed')
                ->where('plan_id', $plan->id)
                ->sum('amount');
        }

        return view('admin.subscriptions.index', compact('payments', 'stats', 'chartData'));
    }

    /**
     * Approve a pending payment
     */
    public function approve(Payment $payment)
    {
        $payment->update([
            'status' => 'completed',
            'paid_at' => now(),
        ]);

        // Update user's plan
        if ($payment->plan_id) {
            $payment->user->update(['plan_id' => $payment->plan_id]);
        }

        // Create or update subscription
        Subscription::updateOrCreate(
            ['user_id' => $payment->user_id, 'status' => 'active'],
            [
                'plan_id' => $payment->plan_id,
                'amount' => $payment->amount,
                'payment_method' => $payment->payment_method,
                'starts_at' => now(),
                'ends_at' => now()->addMonth(),
            ]
        );

        return back()->with('success', 'Payment approved successfully.');
    }

    /**
     * Reject a pending payment
     */
    public function reject(Payment $payment)
    {
        $payment->update(['status' => 'failed']);

        return back()->with('success', 'Payment rejected.');
    }
}
