<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SupportTicket;
use App\Models\SupportMessage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SupportController extends Controller
{
    /**
     * List all support tickets
     */
    public function index(Request $request)
    {
        $query = SupportTicket::with(['user', 'assignedTo'])
            ->latest('last_reply_at');

        // Filter by status
        if ($request->status && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        // Filter by priority
        if ($request->priority) {
            $query->where('priority', $request->priority);
        }

        // Filter by category
        if ($request->category) {
            $query->where('category', $request->category);
        }

        $tickets = $query->paginate(20);

        // Stats
        $stats = [
            'open' => SupportTicket::where('status', 'open')->count(),
            'in_progress' => SupportTicket::where('status', 'in_progress')->count(),
            'waiting' => SupportTicket::where('status', 'waiting')->count(),
            'urgent' => SupportTicket::where('priority', 'urgent')->where('status', '!=', 'closed')->count(),
        ];

        return view('admin.support.index', compact('tickets', 'stats'));
    }

    /**
     * Show a specific ticket
     */
    public function show(SupportTicket $ticket)
    {
        $ticket->load(['user', 'messages.user', 'assignedTo']);

        return view('admin.support.show', compact('ticket'));
    }

    /**
     * Reply to a ticket
     */
    public function reply(Request $request, SupportTicket $ticket)
    {
        $request->validate([
            'message' => 'required|string|min:5',
            'status' => 'nullable|in:open,in_progress,waiting,resolved,closed',
        ]);

        SupportMessage::create([
            'ticket_id' => $ticket->id,
            'user_id' => Auth::id(),
            'message' => $request->message,
            'is_admin_reply' => true,
        ]);

        $updateData = ['last_reply_at' => now()];
        
        if ($request->status) {
            $updateData['status'] = $request->status;
            if ($request->status === 'resolved') {
                $updateData['resolved_at'] = now();
            }
        }

        if (!$ticket->assigned_to) {
            $updateData['assigned_to'] = Auth::id();
        }

        $ticket->update($updateData);

        return back()->with('success', 'Reply sent successfully!');
    }

    /**
     * Update ticket status
     */
    public function updateStatus(Request $request, SupportTicket $ticket)
    {
        $request->validate([
            'status' => 'required|in:open,in_progress,waiting,resolved,closed',
        ]);

        $updateData = ['status' => $request->status];
        
        if ($request->status === 'resolved') {
            $updateData['resolved_at'] = now();
        }

        $ticket->update($updateData);

        return back()->with('success', 'Ticket status updated.');
    }

    /**
     * Assign ticket to admin
     */
    public function assign(Request $request, SupportTicket $ticket)
    {
        $ticket->update([
            'assigned_to' => Auth::id(),
            'status' => 'in_progress',
        ]);

        return back()->with('success', 'Ticket assigned to you.');
    }
}
