<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Exception;

class InstallController extends Controller
{
    /**
     * Check if app is already installed
     */
    public function __construct()
    {
        if ($this->isInstalled() && !request()->routeIs('install.complete')) {
            abort(404);
        }
    }

    /**
     * Step 1: Welcome & Requirements Check
     */
    public function welcome()
    {
        $requirements = $this->checkRequirements();
        $allPassed = !in_array(false, array_column($requirements, 'passed'));
        
        return view('install.welcome', compact('requirements', 'allPassed'));
    }

    /**
     * Step 2: Database Configuration
     */
    public function database()
    {
        return view('install.database');
    }

    /**
     * Step 2: Save Database Configuration
     */
    public function saveDatabase(Request $request)
    {
        $request->validate([
            'db_host' => 'required',
            'db_port' => 'required|numeric',
            'db_database' => 'required',
            'db_username' => 'required',
            'db_password' => 'nullable',
        ]);

        // Test database connection
        try {
            $pdo = new \PDO(
                "mysql:host={$request->db_host};port={$request->db_port};dbname={$request->db_database}",
                $request->db_username,
                $request->db_password
            );
            $pdo->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
        } catch (Exception $e) {
            return back()->withErrors(['database' => 'Database connection failed: ' . $e->getMessage()]);
        }

        // Save to session for next step
        session([
            'install_db' => [
                'host' => $request->db_host,
                'port' => $request->db_port,
                'database' => $request->db_database,
                'username' => $request->db_username,
                'password' => $request->db_password ?? '',
            ]
        ]);

        return redirect()->route('install.admin');
    }

    /**
     * Step 3: Admin Account Setup
     */
    public function admin()
    {
        if (!session('install_db')) {
            return redirect()->route('install.database');
        }
        
        return view('install.admin');
    }

    /**
     * Step 3: Create Admin & Finalize Installation
     */
    public function saveAdmin(Request $request)
    {
        $request->validate([
            'admin_name' => 'required|string|max:255',
            'admin_email' => 'required|email',
            'admin_password' => 'required|min:6|confirmed',
            'site_name' => 'required|string|max:255',
            'site_url' => 'required|url',
        ]);

        $dbConfig = session('install_db');
        if (!$dbConfig) {
            return redirect()->route('install.database');
        }

        try {
            // Update .env file
            $this->updateEnvFile([
                'APP_NAME' => '"' . $request->site_name . '"',
                'APP_URL' => $request->site_url,
                'DB_HOST' => $dbConfig['host'],
                'DB_PORT' => $dbConfig['port'],
                'DB_DATABASE' => $dbConfig['database'],
                'DB_USERNAME' => $dbConfig['username'],
                'DB_PASSWORD' => '"' . $dbConfig['password'] . '"',
                'QUEUE_CONNECTION' => 'database',
                'SESSION_DRIVER' => 'database',
            ]);

            // Clear config cache
            Artisan::call('config:clear');
            
            // Reconnect with new credentials
            config([
                'database.connections.mysql.host' => $dbConfig['host'],
                'database.connections.mysql.port' => $dbConfig['port'],
                'database.connections.mysql.database' => $dbConfig['database'],
                'database.connections.mysql.username' => $dbConfig['username'],
                'database.connections.mysql.password' => $dbConfig['password'],
            ]);
            
            DB::purge('mysql');
            DB::reconnect('mysql');

            // Run migrations
            Artisan::call('migrate:fresh', ['--force' => true]);

            // Seed plans and settings
            Artisan::call('db:seed', ['--class' => 'PlanSeeder', '--force' => true]);
            Artisan::call('db:seed', ['--class' => 'SettingSeeder', '--force' => true]);

            // Create admin user
            DB::table('users')->insert([
                'name' => $request->admin_name,
                'email' => $request->admin_email,
                'password' => Hash::make($request->admin_password),
                'role' => 'admin',
                'plan_id' => 3,
                'is_active' => true,
                'email_verified_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Create installed flag
            File::put(storage_path('installed'), date('Y-m-d H:i:s'));

            // Clear session
            session()->forget('install_db');

            return redirect()->route('install.complete');

        } catch (Exception $e) {
            return back()->withErrors(['install' => 'Installation failed: ' . $e->getMessage()]);
        }
    }

    /**
     * Installation Complete
     */
    public function complete()
    {
        return view('install.complete');
    }

    /**
     * Check system requirements
     */
    protected function checkRequirements(): array
    {
        return [
            [
                'name' => 'PHP Version',
                'required' => '8.2+',
                'current' => PHP_VERSION,
                'passed' => version_compare(PHP_VERSION, '8.2.0', '>='),
            ],
            [
                'name' => 'PDO Extension',
                'required' => 'Enabled',
                'current' => extension_loaded('pdo') ? 'Enabled' : 'Disabled',
                'passed' => extension_loaded('pdo'),
            ],
            [
                'name' => 'PDO MySQL',
                'required' => 'Enabled',
                'current' => extension_loaded('pdo_mysql') ? 'Enabled' : 'Disabled',
                'passed' => extension_loaded('pdo_mysql'),
            ],
            [
                'name' => 'OpenSSL Extension',
                'required' => 'Enabled',
                'current' => extension_loaded('openssl') ? 'Enabled' : 'Disabled',
                'passed' => extension_loaded('openssl'),
            ],
            [
                'name' => 'Mbstring Extension',
                'required' => 'Enabled',
                'current' => extension_loaded('mbstring') ? 'Enabled' : 'Disabled',
                'passed' => extension_loaded('mbstring'),
            ],
            [
                'name' => 'Tokenizer Extension',
                'required' => 'Enabled',
                'current' => extension_loaded('tokenizer') ? 'Enabled' : 'Disabled',
                'passed' => extension_loaded('tokenizer'),
            ],
            [
                'name' => 'Fileinfo Extension',
                'required' => 'Enabled',
                'current' => extension_loaded('fileinfo') ? 'Enabled' : 'Disabled',
                'passed' => extension_loaded('fileinfo'),
            ],
            [
                'name' => 'Storage Writable',
                'required' => 'Writable',
                'current' => is_writable(storage_path()) ? 'Writable' : 'Not Writable',
                'passed' => is_writable(storage_path()),
            ],
            [
                'name' => 'Bootstrap Cache Writable',
                'required' => 'Writable',
                'current' => is_writable(base_path('bootstrap/cache')) ? 'Writable' : 'Not Writable',
                'passed' => is_writable(base_path('bootstrap/cache')),
            ],
            [
                'name' => '.env File Writable',
                'required' => 'Writable',
                'current' => is_writable(base_path('.env')) ? 'Writable' : 'Not Writable',
                'passed' => is_writable(base_path('.env')),
            ],
        ];
    }

    /**
     * Check if application is installed
     */
    protected function isInstalled(): bool
    {
        return File::exists(storage_path('installed'));
    }

    /**
     * Update .env file values
     */
    protected function updateEnvFile(array $data): void
    {
        $envPath = base_path('.env');
        $envContent = File::get($envPath);

        foreach ($data as $key => $value) {
            $pattern = "/^{$key}=.*/m";
            $replacement = "{$key}={$value}";
            
            if (preg_match($pattern, $envContent)) {
                $envContent = preg_replace($pattern, $replacement, $envContent);
            } else {
                $envContent .= "\n{$key}={$value}";
            }
        }

        File::put($envPath, $envContent);
    }
}
