<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;

class ProfileController extends Controller
{
    /**
     * Show user profile page
     */
    public function index()
    {
        $user = Auth::user();
        $user->load(['plan', 'socialAccounts', 'schedules']);
        
        $stats = [
            'total_videos' => $user->videos()->count(),
            'uploaded_videos' => $user->videos()->where('is_uploaded', true)->count(),
            'pending_videos' => $user->videos()->where('is_uploaded', false)->count(),
            'total_schedules' => $user->schedules()->count(),
            'active_schedules' => $user->schedules()->where('status', 'active')->count(),
        ];

        return view('profile.index', compact('user', 'stats'));
    }

    /**
     * Update profile info
     */
    public function update(Request $request)
    {
        $user = Auth::user();
        
        $request->validate([
            'name' => 'required|string|max:255',
        ]);

        $user->update(['name' => $request->name]);

        return back()->with('success', 'Profile updated successfully.');
    }

    /**
     * Update email
     */
    public function updateEmail(Request $request)
    {
        $user = Auth::user();
        
        $request->validate([
            'email' => 'required|email|unique:users,email,' . $user->id,
            'current_password' => 'required',
        ]);

        if (!Hash::check($request->current_password, $user->password)) {
            return back()->withErrors(['current_password' => 'Current password is incorrect.']);
        }

        $user->update([
            'email' => $request->email,
            'email_verified_at' => null, // Require re-verification
        ]);

        return back()->with('success', 'Email updated. Please verify your new email.');
    }

    /**
     * Update password
     */
    public function updatePassword(Request $request)
    {
        $user = Auth::user();
        
        $request->validate([
            'current_password' => 'required',
            'password' => ['required', 'confirmed', Password::min(6)],
        ]);

        if (!Hash::check($request->current_password, $user->password)) {
            return back()->withErrors(['current_password' => 'Current password is incorrect.']);
        }

        $user->update([
            'password' => Hash::make($request->password),
        ]);

        return back()->with('success', 'Password updated successfully.');
    }
}
