<?php

namespace App\Http\Controllers;

use App\Models\Schedule;
use App\Models\SocialAccount;
use App\Models\Video;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class ScheduleController extends Controller
{
    public function index()
    {
        $schedules = Auth::user()->schedules()
            ->with(['socialAccount', 'videos'])
            ->latest()
            ->paginate(10);

        return view('schedules.index', compact('schedules'));
    }

    public function create()
    {
        $accounts = Auth::user()->socialAccounts()
            ->where('needs_reauth', false)
            ->get();

        $plan = Auth::user()->plan;

        return view('schedules.create', compact('accounts', 'plan'));
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        $plan = $user->plan;

        // Calculate max file size in KB for validation (plan stores MB)
        $maxVideoSizeKb = ($plan->max_video_size_mb ?? 100) * 1024;

        $request->validate([
            'social_account_id' => 'required|exists:social_accounts,id',
            'name' => 'nullable|string|max:255',
            'upload_mode' => 'required|in:normal,shorts',
            'interval_minutes' => "required|integer|min:{$plan->min_interval_minutes}",
            'auto_restart' => 'boolean',
            'title_template' => 'required|string|max:255',
            'description_template' => 'nullable|string',
            'videos' => "required|array|min:1|max:{$plan->max_videos_per_batch}",
            'videos.*' => "file|mimetypes:video/*|max:{$maxVideoSizeKb}",
        ], [
            'videos.*.max' => "Each video must be less than {$plan->formatted_max_video_size}. Upgrade your plan for larger files.",
        ]);

        // Verify the social account belongs to the user
        $account = SocialAccount::where('id', $request->social_account_id)
            ->where('user_id', $user->id)
            ->firstOrFail();

        // Create the schedule
        $schedule = Schedule::create([
            'user_id' => $user->id,
            'social_account_id' => $account->id,
            'name' => $request->name ?? 'Schedule ' . now()->format('M d, Y'),
            'upload_mode' => $request->upload_mode,
            'interval_minutes' => $request->interval_minutes,
            'auto_restart' => $request->boolean('auto_restart', true),
            'status' => 'active',
            'next_run_at' => now()->addMinutes($request->interval_minutes),
        ]);

        // Store videos
        $sortOrder = 0;
        foreach ($request->file('videos') as $videoFile) {
            $path = $videoFile->store('videos/' . $user->id, 'private');
            
            // Generate title with {index}
            $title = str_replace('{index}', $sortOrder + 1, $request->title_template);
            $description = str_replace('{index}', $sortOrder + 1, $request->description_template ?? '');

            Video::create([
                'schedule_id' => $schedule->id,
                'user_id' => $user->id,
                'file_path' => $path,
                'original_filename' => $videoFile->getClientOriginalName(),
                'file_size' => $videoFile->getSize(),
                'title' => $title,
                'description' => $description,
                'sort_order' => $sortOrder,
            ]);

            $sortOrder++;
        }

        return redirect()->route('schedules.index')
            ->with('success', "Schedule created with {$sortOrder} videos!");
    }

    public function show(Schedule $schedule)
    {
        $this->authorize('view', $schedule);
        
        $schedule->load(['socialAccount', 'videos']);
        
        return view('schedules.show', compact('schedule'));
    }

    public function pause(Schedule $schedule)
    {
        $this->authorize('update', $schedule);
        
        $schedule->update([
            'status' => 'paused',
            'next_run_at' => null,
        ]);

        return back()->with('success', 'Schedule paused.');
    }

    public function resume(Schedule $schedule)
    {
        $this->authorize('update', $schedule);
        
        if ($schedule->pendingVideos()->count() === 0) {
            return back()->with('error', 'No pending videos to resume.');
        }

        $schedule->update([
            'status' => 'active',
            'next_run_at' => now(),
        ]);

        return back()->with('success', 'Schedule resumed.');
    }

    public function destroy(Schedule $schedule)
    {
        $this->authorize('delete', $schedule);
        
        // Delete video files
        foreach ($schedule->videos as $video) {
            Storage::disk('private')->delete($video->file_path);
        }

        $schedule->delete();

        return redirect()->route('schedules.index')
            ->with('success', 'Schedule deleted.');
    }
}
