<?php

namespace App\Http\Controllers;

use App\Models\SocialAccount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Laravel\Socialite\Facades\Socialite;
use Exception;

class SocialAccountController extends Controller
{
    /**
     * Show connected accounts
     */
    public function index()
    {
        $accounts = Auth::user()->socialAccounts()->get();
        
        // Available platforms with connection status
        $platforms = [
            'youtube' => [
                'name' => 'YouTube',
                'icon' => 'fab fa-youtube',
                'color' => 'danger',
                'configured' => !empty(config('services.google.client_id')),
                'connected' => $accounts->where('platform', 'youtube')->first(),
            ],
            'facebook' => [
                'name' => 'Facebook',
                'icon' => 'fab fa-facebook',
                'color' => 'primary',
                'configured' => !empty(config('services.facebook.client_id')),
                'connected' => $accounts->where('platform', 'facebook')->first(),
            ],
            'tiktok' => [
                'name' => 'TikTok',
                'icon' => 'fab fa-tiktok',
                'color' => 'dark',
                'configured' => !empty(config('services.tiktok.client_key')),
                'connected' => $accounts->where('platform', 'tiktok')->first(),
            ],
        ];
        
        return view('accounts.index', compact('accounts', 'platforms'));
    }

    /**
     * Redirect to YouTube OAuth
     */
    public function redirectToYouTube()
    {
        if (empty(config('services.google.client_id'))) {
            return back()->with('error', 'YouTube integration is not configured. Please contact admin.');
        }
        
        return Socialite::driver('google')
            ->scopes([
                'https://www.googleapis.com/auth/youtube.upload',
                'https://www.googleapis.com/auth/youtube.readonly',
            ])
            ->with([
                'access_type' => 'offline',
                'prompt' => 'consent',
            ])
            ->redirect();
    }

    /**
     * Handle YouTube OAuth callback
     */
    public function handleYouTubeCallback()
    {
        try {
            $googleUser = Socialite::driver('google')->user();

            SocialAccount::updateOrCreate(
                [
                    'user_id' => Auth::id(),
                    'platform' => 'youtube',
                    'platform_user_id' => $googleUser->getId(),
                ],
                [
                    'platform_user_name' => $googleUser->getName(),
                    'token' => encrypt($googleUser->token),
                    'refresh_token' => $googleUser->refreshToken ? encrypt($googleUser->refreshToken) : null,
                    'expires_at' => $googleUser->expiresIn ? now()->addSeconds($googleUser->expiresIn) : null,
                    'needs_reauth' => false,
                ]
            );

            return redirect()->route('accounts.index')
                ->with('success', 'YouTube channel connected successfully!');

        } catch (Exception $e) {
            return redirect()->route('accounts.index')
                ->with('error', 'Failed to connect YouTube: ' . $e->getMessage());
        }
    }

    /**
     * Redirect to Facebook OAuth
     */
    public function redirectToFacebook()
    {
        if (empty(config('services.facebook.client_id'))) {
            return back()->with('error', 'Facebook integration is not configured. Please contact admin.');
        }
        
        return Socialite::driver('facebook')
            ->scopes([
                'pages_manage_posts',
                'pages_read_engagement',
                'pages_show_list',
                'publish_video',
            ])
            ->redirect();
    }

    /**
     * Handle Facebook OAuth callback
     */
    public function handleFacebookCallback()
    {
        try {
            $facebookUser = Socialite::driver('facebook')->user();

            // Get user's pages with video publish access
            $response = Http::get('https://graph.facebook.com/v18.0/me/accounts', [
                'access_token' => $facebookUser->token,
                'fields' => 'id,name,access_token',
            ]);

            $pages = $response->json('data', []);
            
            if (empty($pages)) {
                return redirect()->route('accounts.index')
                    ->with('error', 'No Facebook Pages found. Please create a Page first.');
            }

            // Store first page (user can change later)
            $page = $pages[0];
            
            SocialAccount::updateOrCreate(
                [
                    'user_id' => Auth::id(),
                    'platform' => 'facebook',
                    'platform_user_id' => $page['id'],
                ],
                [
                    'platform_user_name' => $page['name'],
                    'token' => encrypt($page['access_token']),
                    'refresh_token' => null, // Page tokens don't expire
                    'expires_at' => null,
                    'needs_reauth' => false,
                ]
            );

            return redirect()->route('accounts.index')
                ->with('success', 'Facebook Page connected successfully!');

        } catch (Exception $e) {
            return redirect()->route('accounts.index')
                ->with('error', 'Failed to connect Facebook: ' . $e->getMessage());
        }
    }

    /**
     * Redirect to TikTok OAuth
     */
    public function redirectToTikTok()
    {
        if (empty(config('services.tiktok.client_key'))) {
            return back()->with('error', 'TikTok integration is not configured. Please contact admin.');
        }
        
        $clientKey = config('services.tiktok.client_key');
        $redirectUri = urlencode(config('services.tiktok.redirect'));
        $scope = urlencode('user.info.basic,video.upload,video.publish');
        $state = csrf_token();
        
        session(['tiktok_state' => $state]);
        
        $url = "https://www.tiktok.com/v2/auth/authorize/"
             . "?client_key={$clientKey}"
             . "&response_type=code"
             . "&scope={$scope}"
             . "&redirect_uri={$redirectUri}"
             . "&state={$state}";
             
        return redirect($url);
    }

    /**
     * Handle TikTok OAuth callback
     */
    public function handleTikTokCallback(Request $request)
    {
        try {
            // Verify state
            if ($request->state !== session('tiktok_state')) {
                throw new Exception('Invalid state parameter');
            }
            
            if ($request->has('error')) {
                throw new Exception($request->error_description ?? 'OAuth cancelled');
            }
            
            $code = $request->code;
            
            // Exchange code for access token
            $response = Http::asForm()->post('https://open.tiktokapis.com/v2/oauth/token/', [
                'client_key' => config('services.tiktok.client_key'),
                'client_secret' => config('services.tiktok.client_secret'),
                'code' => $code,
                'grant_type' => 'authorization_code',
                'redirect_uri' => config('services.tiktok.redirect'),
            ]);
            
            if (!$response->successful()) {
                throw new Exception('Failed to get access token');
            }
            
            $data = $response->json();
            $accessToken = $data['access_token'];
            $openId = $data['open_id'];
            $expiresIn = $data['expires_in'] ?? 86400;
            $refreshToken = $data['refresh_token'] ?? null;
            
            // Get user info
            $userResponse = Http::withHeaders([
                'Authorization' => 'Bearer ' . $accessToken,
            ])->get('https://open.tiktokapis.com/v2/user/info/', [
                'fields' => 'open_id,display_name,avatar_url',
            ]);
            
            $userInfo = $userResponse->json('data.user', []);
            $displayName = $userInfo['display_name'] ?? 'TikTok User';
            
            SocialAccount::updateOrCreate(
                [
                    'user_id' => Auth::id(),
                    'platform' => 'tiktok',
                    'platform_user_id' => $openId,
                ],
                [
                    'platform_user_name' => $displayName,
                    'token' => encrypt($accessToken),
                    'refresh_token' => $refreshToken ? encrypt($refreshToken) : null,
                    'expires_at' => now()->addSeconds($expiresIn),
                    'needs_reauth' => false,
                ]
            );

            return redirect()->route('accounts.index')
                ->with('success', 'TikTok account connected successfully!');

        } catch (Exception $e) {
            return redirect()->route('accounts.index')
                ->with('error', 'Failed to connect TikTok: ' . $e->getMessage());
        }
    }

    /**
     * Disconnect a social account
     */
    public function disconnect(SocialAccount $account)
    {
        if ($account->user_id !== Auth::id()) {
            abort(403);
        }

        $activeSchedules = $account->schedules()->where('status', 'active')->count();
        if ($activeSchedules > 0) {
            return back()->with('error', 'Cannot disconnect: This account has active schedules. Please pause or delete them first.');
        }

        $platform = ucfirst($account->platform);
        $account->delete();

        return back()->with('success', "{$platform} account disconnected successfully.");
    }
}
