<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use App\Models\PaymentGateway;
use App\Models\Plan;
use App\Models\Subscription;
use App\Models\User;
use App\Notifications\NewPurchaseNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;

class SubscriptionController extends Controller
{
    /**
     * Show subscription/billing page
     */
    public function index()
    {
        $user = Auth::user();
        $user->load('plan');
        
        $plans = Plan::orderBy('price')->get();
        $currentSubscription = Subscription::where('user_id', $user->id)
            ->where('status', 'active')
            ->first();
        
        $paymentHistory = Payment::where('user_id', $user->id)
            ->with('plan')
            ->latest()
            ->limit(10)
            ->get();
        
        $gateways = PaymentGateway::getActive();

        return view('subscription.index', compact(
            'user', 'plans', 'currentSubscription', 'paymentHistory', 'gateways'
        ));
    }

    /**
     * Initiate upgrade process
     */
    public function upgrade(Request $request)
    {
        $request->validate([
            'plan_id' => 'required|exists:plans,id',
            'payment_method' => 'required|string',
        ]);

        $user = Auth::user();
        $plan = Plan::findOrFail($request->plan_id);
        $gateway = PaymentGateway::where('name', $request->payment_method)->first();

        if (!$gateway || !$gateway->is_active) {
            return back()->with('error', 'Selected payment method is not available.');
        }

        // Create pending payment
        $payment = Payment::create([
            'user_id' => $user->id,
            'plan_id' => $plan->id,
            'payment_method' => $request->payment_method,
            'amount' => $plan->price,
            'currency' => 'USD',
            'status' => 'pending',
        ]);

        // Notify admins about new pending payment
        $admins = User::where('role', 'admin')->get();
        Notification::send($admins, new NewPurchaseNotification($payment));

        // Handle different payment methods
        if ($request->payment_method === 'manual') {
            return redirect()->route('subscription.manual', $payment);
        }

        // For other gateways, redirect to payment page (simplified)
        return redirect()->route('subscription.pay', ['payment' => $payment, 'gateway' => $request->payment_method]);
    }

    /**
     * Show manual payment instructions
     */
    public function manual(Payment $payment)
    {
        if ($payment->user_id !== Auth::id()) {
            abort(403);
        }

        $gateway = PaymentGateway::where('name', 'manual')->first();
        
        return view('subscription.manual', compact('payment', 'gateway'));
    }

    /**
     * Generic payment page (for gateway redirects)
     */
    public function pay(Payment $payment, string $gateway)
    {
        if ($payment->user_id !== Auth::id()) {
            abort(403);
        }

        $gatewayModel = PaymentGateway::where('name', $gateway)->first();
        
        return view('subscription.pay', compact('payment', 'gatewayModel'));
    }

    /**
     * Cancel subscription
     */
    public function cancel(Request $request)
    {
        $user = Auth::user();
        
        $subscription = Subscription::where('user_id', $user->id)
            ->where('status', 'active')
            ->first();

        if ($subscription) {
            $subscription->update(['status' => 'cancelled']);
            
            // Downgrade to free plan
            $freePlan = Plan::where('price', 0)->first();
            if ($freePlan) {
                $user->update(['plan_id' => $freePlan->id]);
            }
        }

        return back()->with('success', 'Subscription cancelled. You\'ve been moved to the free plan.');
    }
}
