<?php

namespace App\Http\Controllers;

use App\Models\SupportTicket;
use App\Models\SupportMessage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SupportController extends Controller
{
    /**
     * List user's support tickets
     */
    public function index()
    {
        $tickets = Auth::user()->supportTickets()
            ->with('messages')
            ->latest()
            ->paginate(10);

        return view('support.index', compact('tickets'));
    }

    /**
     * Show create ticket form
     */
    public function create()
    {
        return view('support.create');
    }

    /**
     * Store a new support ticket
     */
    public function store(Request $request)
    {
        $request->validate([
            'subject' => 'required|string|max:255',
            'category' => 'required|in:technical,billing,account,feature,other',
            'priority' => 'required|in:low,medium,high,urgent',
            'message' => 'required|string|min:10',
        ]);

        $ticket = SupportTicket::create([
            'user_id' => Auth::id(),
            'subject' => $request->subject,
            'category' => $request->category,
            'priority' => $request->priority,
            'last_reply_at' => now(),
        ]);

        SupportMessage::create([
            'ticket_id' => $ticket->id,
            'user_id' => Auth::id(),
            'message' => $request->message,
            'is_admin_reply' => false,
        ]);

        return redirect()->route('support.show', $ticket)
            ->with('success', "Ticket {$ticket->ticket_number} created successfully!");
    }

    /**
     * Show a specific ticket
     */
    public function show(SupportTicket $ticket)
    {
        // Ensure user owns this ticket
        if ($ticket->user_id !== Auth::id()) {
            abort(403);
        }

        $ticket->load(['messages.user', 'assignedTo']);

        return view('support.show', compact('ticket'));
    }

    /**
     * Reply to a ticket
     */
    public function reply(Request $request, SupportTicket $ticket)
    {
        if ($ticket->user_id !== Auth::id()) {
            abort(403);
        }

        if (in_array($ticket->status, ['closed', 'resolved'])) {
            return back()->with('error', 'Cannot reply to a closed ticket.');
        }

        $request->validate([
            'message' => 'required|string|min:5',
        ]);

        SupportMessage::create([
            'ticket_id' => $ticket->id,
            'user_id' => Auth::id(),
            'message' => $request->message,
            'is_admin_reply' => false,
        ]);

        $ticket->update([
            'last_reply_at' => now(),
            'status' => 'open', // Reopen if was waiting
        ]);

        return back()->with('success', 'Reply sent successfully!');
    }

    /**
     * Close a ticket
     */
    public function close(SupportTicket $ticket)
    {
        if ($ticket->user_id !== Auth::id()) {
            abort(403);
        }

        $ticket->update([
            'status' => 'closed',
            'resolved_at' => now(),
        ]);

        return back()->with('success', 'Ticket closed.');
    }
}
