<?php

namespace App\Jobs;

use App\Models\Schedule;
use App\Models\UploadLog;
use App\Models\Video;
use App\Services\YouTubeUploadService;
use App\Services\FacebookUploadService;
use App\Services\TikTokUploadService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessVideoUpload implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $backoff = 60;

    public function __construct(
        public Video $video,
        public Schedule $schedule
    ) {}

    public function handle(): void
    {
        $video = $this->video;
        $schedule = $this->schedule->fresh();

        // Check if schedule is still active
        if (!in_array($schedule->status, ['active'])) {
            Log::info("Schedule {$schedule->id} is not active, skipping upload");
            return;
        }

        // Check if video is already uploaded
        if ($video->is_uploaded) {
            Log::info("Video {$video->id} already uploaded, skipping");
            return;
        }

        // Check user's daily limit
        $user = $schedule->user;
        if ($user->hasReachedDailyLimit()) {
            $this->handleQuotaExceeded('User daily limit reached');
            return;
        }

        // Get the appropriate upload service based on platform
        $account = $schedule->socialAccount;
        $platform = $account->platform;
        
        $result = $this->uploadToPlatform($video, $account, $platform);

        if ($result['success']) {
            $this->handleSuccess($result['video_id']);
        } else {
            $this->handleFailure($result);
        }
    }

    /**
     * Upload to the appropriate platform
     */
    protected function uploadToPlatform(Video $video, $account, string $platform): array
    {
        return match($platform) {
            'youtube' => $this->uploadToYouTube($video, $account),
            'facebook' => $this->uploadToFacebook($video, $account),
            'tiktok' => $this->uploadToTikTok($video, $account),
            default => [
                'success' => false,
                'error' => 'unsupported_platform',
                'message' => "Platform {$platform} is not supported",
            ],
        };
    }

    protected function uploadToYouTube(Video $video, $account): array
    {
        $service = app(YouTubeUploadService::class);
        
        if (empty(config('services.google.client_id'))) {
            return $service->simulateUpload($video);
        }
        
        return $service->upload($video, $account);
    }

    protected function uploadToFacebook(Video $video, $account): array
    {
        $service = app(FacebookUploadService::class);
        
        if (empty(config('services.facebook.client_id'))) {
            return $service->simulateUpload($video);
        }
        
        return $service->upload($video, $account);
    }

    protected function uploadToTikTok(Video $video, $account): array
    {
        $service = app(TikTokUploadService::class);
        
        if (empty(config('services.tiktok.client_key'))) {
            return $service->simulateUpload($video);
        }
        
        return $service->upload($video, $account);
    }

    /**
     * Handle successful upload
     */
    protected function handleSuccess(string $platformVideoId): void
    {
        // Mark video as uploaded
        $this->video->markAsUploaded($platformVideoId);

        // Increment daily count
        $this->schedule->increment('daily_uploaded_count');

        // Log success
        UploadLog::logSuccess($this->video);

        // Schedule next video
        $this->schedule->scheduleNextUpload();

        Log::info("Video {$this->video->id} uploaded successfully: {$platformVideoId}");
    }

    /**
     * Handle upload failure
     */
    protected function handleFailure(array $result): void
    {
        $error = $result['error'] ?? 'unknown';
        $message = $result['message'] ?? 'Unknown error';

        // Mark video as failed
        $this->video->markAsFailed($message);

        // Log failure
        UploadLog::logFailure($this->video, $message, $result);

        // Handle quota exceeded
        if ($error === 'quota_exceeded') {
            $this->handleQuotaExceeded($message);
            return;
        }

        // Handle auth error - pause schedule
        if ($error === 'auth_error') {
            $this->schedule->update(['status' => 'paused']);
            UploadLog::create([
                'user_id' => $this->schedule->user_id,
                'schedule_id' => $this->schedule->id,
                'status' => 'auth_error',
                'message' => 'Schedule paused due to authentication error',
            ]);
            return;
        }

        // For other errors, retry if attempts remaining
        if ($this->video->upload_attempts >= 3) {
            Log::warning("Video {$this->video->id} failed after 3 attempts, skipping");
            $this->schedule->scheduleNextUpload();
        }
    }

    /**
     * Handle quota exceeded - pause and set resume time
     */
    protected function handleQuotaExceeded(string $message): void
    {
        Log::warning("Quota exceeded for schedule {$this->schedule->id}: {$message}");

        // Pause schedule for 24 hours
        $this->schedule->pauseForQuota();

        // Log quota exceeded
        UploadLog::logQuotaExceeded($this->schedule, $message);
    }

    /**
     * Handle job failure
     */
    public function failed(\Throwable $exception): void
    {
        Log::error("Job failed for video {$this->video->id}: " . $exception->getMessage());
        
        $this->video->markAsFailed($exception->getMessage());
        
        UploadLog::logFailure($this->video, 'Job failed: ' . $exception->getMessage());
    }
}
