<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;

class AdPlacement extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'display_name',
        'location',
        'ad_code',
        'is_active',
        'size',
        'show_to',
        'sort_order',
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    /**
     * Get active ads for a specific location
     */
    public static function getAdsForLocation(string $location): \Illuminate\Database\Eloquent\Collection
    {
        return Cache::remember("ads_{$location}", 300, function () use ($location) {
            return static::where('location', $location)
                ->where('is_active', true)
                ->whereNotNull('ad_code')
                ->orderBy('sort_order')
                ->get();
        });
    }

    /**
     * Check if this ad should be shown to current user
     */
    public function shouldShow(): bool
    {
        if (!$this->is_active || empty($this->ad_code)) {
            return false;
        }

        $user = Auth::user();

        return match($this->show_to) {
            'guests' => !$user,
            'users' => (bool)$user,
            'free_only' => $user && $user->plan && $user->plan->price == 0,
            default => true, // 'all'
        };
    }

    /**
     * Render the ad HTML
     */
    public function render(): string
    {
        if (!$this->shouldShow()) {
            return '';
        }

        return '<div class="ad-placement ad-' . e($this->name) . '" data-ad-id="' . $this->id . '">' 
             . $this->ad_code 
             . '</div>';
    }

    /**
     * Clear all ad caches
     */
    public static function clearCache(): void
    {
        $locations = ['header', 'sidebar', 'footer', 'content', 'dashboard'];
        foreach ($locations as $location) {
            Cache::forget("ads_{$location}");
        }
    }
}
