<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Plan extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'max_videos_per_batch',
        'min_interval_minutes',
        'daily_upload_cap',
        'max_video_size_mb',
        'max_storage_mb',
        'price_id',
        'price',
        'features',
    ];

    protected $casts = [
        'features' => 'array',
        'price' => 'decimal:2',
    ];

    public function users(): HasMany
    {
        return $this->hasMany(User::class);
    }

    /**
     * Get max video size in bytes
     */
    public function getMaxVideoSizeBytesAttribute(): int
    {
        return ($this->max_video_size_mb ?? 100) * 1024 * 1024;
    }

    /**
     * Get formatted max video size (e.g., "100 MB" or "2 GB")
     */
    public function getFormattedMaxVideoSizeAttribute(): string
    {
        $mb = $this->max_video_size_mb ?? 100;
        
        if ($mb >= 1024) {
            return round($mb / 1024, 1) . ' GB';
        }
        
        return $mb . ' MB';
    }

    /**
     * Get formatted max storage (e.g., "1 GB" or "10 GB")
     */
    public function getFormattedMaxStorageAttribute(): string
    {
        $mb = $this->max_storage_mb ?? 1000;
        
        if ($mb >= 1024) {
            return round($mb / 1024, 1) . ' GB';
        }
        
        return $mb . ' MB';
    }

    /**
     * Check if a file size (in bytes) exceeds this plan's limit
     */
    public function exceedsVideoSizeLimit(int $sizeInBytes): bool
    {
        return $sizeInBytes > $this->max_video_size_bytes;
    }
}

