<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Carbon\Carbon;

class Schedule extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'social_account_id',
        'name',
        'upload_mode',
        'interval_minutes',
        'auto_restart',
        'status',
        'daily_uploaded_count',
        'next_run_at',
        'resume_at',
        'last_reset_at',
    ];

    protected $casts = [
        'auto_restart' => 'boolean',
        'next_run_at' => 'datetime',
        'resume_at' => 'datetime',
        'last_reset_at' => 'datetime',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function socialAccount(): BelongsTo
    {
        return $this->belongsTo(SocialAccount::class);
    }

    public function videos(): HasMany
    {
        return $this->hasMany(Video::class)->orderBy('sort_order');
    }

    /**
     * Get pending (not yet uploaded) videos
     */
    public function pendingVideos(): HasMany
    {
        return $this->hasMany(Video::class)
            ->where('is_uploaded', false)
            ->orderBy('sort_order');
    }

    /**
     * Get uploaded videos
     */
    public function uploadedVideos(): HasMany
    {
        return $this->hasMany(Video::class)
            ->where('is_uploaded', true)
            ->orderBy('uploaded_at', 'desc');
    }

    /**
     * Get the next video to upload
     */
    public function getNextVideoAttribute(): ?Video
    {
        return $this->pendingVideos()->first();
    }

    /**
     * Check if schedule is ready to run
     */
    public function isReadyToRun(): bool
    {
        if ($this->status !== 'active') {
            return false;
        }

        if (!$this->next_run_at) {
            return false;
        }

        return $this->next_run_at->isPast();
    }

    /**
     * Check if schedule should resume from quota limit
     */
    public function shouldResume(): bool
    {
        if ($this->status !== 'quota_limit') {
            return false;
        }

        if (!$this->resume_at) {
            return false;
        }

        return $this->resume_at->isPast();
    }

    /**
     * Pause the schedule due to quota limit
     */
    public function pauseForQuota(): void
    {
        $this->update([
            'status' => 'quota_limit',
            'resume_at' => Carbon::now()->addHours(24),
            'next_run_at' => null,
        ]);
    }

    /**
     * Resume the schedule after quota limit
     */
    public function resumeFromQuota(): void
    {
        $this->update([
            'status' => 'active',
            'resume_at' => null,
            'daily_uploaded_count' => 0,
            'last_reset_at' => Carbon::now(),
            'next_run_at' => Carbon::now(),
        ]);
    }

    /**
     * Schedule the next video upload
     */
    public function scheduleNextUpload(): void
    {
        $nextVideo = $this->nextVideo;

        if (!$nextVideo) {
            $this->update(['status' => 'completed', 'next_run_at' => null]);
            return;
        }

        $this->update([
            'next_run_at' => Carbon::now()->addMinutes($this->interval_minutes),
        ]);
    }

    /**
     * Get status badge class for Bootstrap
     */
    public function getStatusBadgeAttribute(): string
    {
        return match($this->status) {
            'active' => 'success',
            'paused' => 'warning',
            'completed' => 'info',
            'quota_limit' => 'danger',
            default => 'secondary',
        };
    }

    /**
     * Pause the schedule manually
     */
    public function pause(): void
    {
        $this->update(['status' => 'paused', 'next_run_at' => null]);
    }

    /**
     * Resume the schedule
     */
    public function resume(): void
    {
        $this->update([
            'status' => 'active',
            'next_run_at' => Carbon::now(),
        ]);
    }

    /**
     * Get upload progress percentage
     */
    public function getProgressAttribute(): int
    {
        $total = $this->videos()->count();
        if ($total === 0) return 0;
        
        $uploaded = $this->videos()->where('is_uploaded', true)->count();
        return (int) round(($uploaded / $total) * 100);
    }
}
