<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class SupportTicket extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'ticket_number',
        'subject',
        'category',
        'priority',
        'status',
        'assigned_to',
        'last_reply_at',
        'resolved_at',
    ];

    protected $casts = [
        'last_reply_at' => 'datetime',
        'resolved_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($ticket) {
            $ticket->ticket_number = 'TKT-' . strtoupper(uniqid());
        });
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function assignedTo(): BelongsTo
    {
        return $this->belongsTo(User::class, 'assigned_to');
    }

    public function messages(): HasMany
    {
        return $this->hasMany(SupportMessage::class, 'ticket_id');
    }

    public function getStatusBadgeAttribute(): string
    {
        return match($this->status) {
            'open' => 'badge-primary',
            'in_progress' => 'badge-info',
            'waiting' => 'badge-warning',
            'resolved' => 'badge-success',
            'closed' => 'badge-secondary',
            default => 'badge-dark',
        };
    }

    public function getPriorityBadgeAttribute(): string
    {
        return match($this->priority) {
            'low' => 'badge-secondary',
            'medium' => 'badge-info',
            'high' => 'badge-warning',
            'urgent' => 'badge-danger',
            default => 'badge-dark',
        };
    }

    public function getCategoryIconAttribute(): string
    {
        return match($this->category) {
            'technical' => 'fas fa-cog',
            'billing' => 'fas fa-credit-card',
            'account' => 'fas fa-user',
            'feature' => 'fas fa-lightbulb',
            default => 'fas fa-question-circle',
        };
    }
}
