<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class UploadLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'schedule_id',
        'video_id',
        'status',
        'message',
        'details',
    ];

    protected $casts = [
        'details' => 'array',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function schedule(): BelongsTo
    {
        return $this->belongsTo(Schedule::class);
    }

    public function video(): BelongsTo
    {
        return $this->belongsTo(Video::class);
    }

    /**
     * Create a success log entry
     */
    public static function logSuccess(Video $video, string $message = 'Video uploaded successfully'): self
    {
        return static::create([
            'user_id' => $video->user_id,
            'schedule_id' => $video->schedule_id,
            'video_id' => $video->id,
            'status' => 'success',
            'message' => $message,
            'details' => [
                'platform_video_id' => $video->platform_video_id,
                'title' => $video->title,
            ],
        ]);
    }

    /**
     * Create a failure log entry
     */
    public static function logFailure(Video $video, string $message, array $details = []): self
    {
        return static::create([
            'user_id' => $video->user_id,
            'schedule_id' => $video->schedule_id,
            'video_id' => $video->id,
            'status' => 'failed',
            'message' => $message,
            'details' => $details,
        ]);
    }

    /**
     * Create a quota exceeded log entry
     */
    public static function logQuotaExceeded(Schedule $schedule, string $message = 'Daily quota exceeded'): self
    {
        return static::create([
            'user_id' => $schedule->user_id,
            'schedule_id' => $schedule->id,
            'status' => 'quota_exceeded',
            'message' => $message,
        ]);
    }

    /**
     * Get status badge class
     */
    public function getStatusBadgeAttribute(): string
    {
        return match($this->status) {
            'success' => 'badge-success',
            'failed' => 'badge-danger',
            'quota_exceeded' => 'badge-warning',
            'auth_error' => 'badge-info',
            default => 'badge-secondary',
        };
    }
}
