<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable
{
    use HasFactory, Notifiable;

    protected $fillable = [
        'name',
        'email',
        'password',
        'role',
        'plan_id',
        'is_active',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'is_active' => 'boolean',
        ];
    }

    public function plan(): BelongsTo
    {
        return $this->belongsTo(Plan::class);
    }

    public function socialAccounts(): HasMany
    {
        return $this->hasMany(SocialAccount::class);
    }

    public function schedules(): HasMany
    {
        return $this->hasMany(Schedule::class);
    }

    public function videos(): HasMany
    {
        return $this->hasMany(Video::class);
    }

    public function uploadLogs(): HasMany
    {
        return $this->hasMany(UploadLog::class);
    }

    public function supportTickets(): HasMany
    {
        return $this->hasMany(SupportTicket::class);
    }

    /**
     * Check if user is admin
     */
    public function isAdmin(): bool
    {
        return $this->role === 'admin';
    }

    /**
     * Get today's upload count across all schedules
     */
    public function getTodayUploadCountAttribute(): int
    {
        return $this->videos()
            ->where('is_uploaded', true)
            ->whereDate('uploaded_at', today())
            ->count();
    }

    /**
     * Get active schedules count
     */
    public function getActiveSchedulesCountAttribute(): int
    {
        return $this->schedules()
            ->where('status', 'active')
            ->count();
    }

    /**
     * Check if user has reached daily upload limit
     */
    public function hasReachedDailyLimit(): bool
    {
        if (!$this->plan) {
            return true;
        }

        return $this->today_upload_count >= $this->plan->daily_upload_cap;
    }

    /**
     * Get the next scheduled upload time
     */
    public function getNextUploadTimeAttribute(): ?\Carbon\Carbon
    {
        $schedule = $this->schedules()
            ->where('status', 'active')
            ->whereNotNull('next_run_at')
            ->orderBy('next_run_at')
            ->first();

        return $schedule?->next_run_at;
    }
}
