<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Video extends Model
{
    use HasFactory;

    protected $fillable = [
        'schedule_id',
        'user_id',
        'file_path',
        'original_filename',
        'file_size',
        'title',
        'description',
        'tags',
        'is_uploaded',
        'upload_attempts',
        'platform_video_id',
        'upload_error',
        'uploaded_at',
        'sort_order',
    ];

    protected $casts = [
        'is_uploaded' => 'boolean',
        'uploaded_at' => 'datetime',
    ];

    public function schedule(): BelongsTo
    {
        return $this->belongsTo(Schedule::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get titled with Shorts hashtag if needed
     */
    public function getTitleForUpload(): string
    {
        $title = $this->title;
        
        if ($this->schedule && $this->schedule->upload_mode === 'shorts') {
            if (stripos($title, '#shorts') === false) {
                $title .= ' #Shorts';
            }
        }
        
        return $title;
    }

    /**
     * Get description with Shorts hashtag if needed
     */
    public function getDescriptionForUpload(): ?string
    {
        $description = $this->description ?? '';
        
        if ($this->schedule && $this->schedule->upload_mode === 'shorts') {
            if (stripos($description, '#shorts') === false) {
                $description .= "\n\n#Shorts";
            }
        }
        
        return $description;
    }

    /**
     * Mark as uploaded successfully
     */
    public function markAsUploaded(string $platformVideoId): void
    {
        $this->update([
            'is_uploaded' => true,
            'platform_video_id' => $platformVideoId,
            'uploaded_at' => now(),
            'upload_error' => null,
        ]);
    }

    /**
     * Mark upload as failed
     */
    public function markAsFailed(string $error): void
    {
        $this->increment('upload_attempts');
        $this->update(['upload_error' => $error]);
    }

    /**
     * Get the file size formatted
     */
    public function getFormattedFileSizeAttribute(): string
    {
        $bytes = $this->file_size;
        $units = ['B', 'KB', 'MB', 'GB'];
        
        for ($i = 0; $bytes > 1024; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }

    /**
     * Get YouTube URL if uploaded
     */
    public function getYoutubeUrlAttribute(): ?string
    {
        if (!$this->platform_video_id) {
            return null;
        }
        
        return "https://youtube.com/watch?v={$this->platform_video_id}";
    }
}
