<?php

namespace App\Services;

use App\Models\SocialAccount;
use App\Models\Video;
use Illuminate\Support\Facades\Http;
use Exception;

class FacebookUploadService
{
    protected string $graphApiVersion = 'v18.0';
    protected string $baseUrl = 'https://graph.facebook.com';

    /**
     * Upload a video to Facebook
     */
    public function upload(Video $video, SocialAccount $account): array
    {
        try {
            // Get access token
            $accessToken = decrypt($account->token);
            $pageId = $account->platform_user_id;

            // Check if account needs reauth
            if ($account->needs_reauth) {
                return [
                    'success' => false,
                    'error' => 'auth_error',
                    'message' => 'Account needs re-authentication',
                ];
            }

            // Get video file path
            $filePath = storage_path('app/private/' . $video->file_path);
            
            if (!file_exists($filePath)) {
                return [
                    'success' => false,
                    'error' => 'file_not_found',
                    'message' => 'Video file not found',
                ];
            }

            $fileSize = filesize($filePath);
            $title = $video->getTitleForUpload();
            $description = $video->getDescriptionForUpload();

            // Step 1: Initialize upload session
            $initResponse = Http::post("{$this->baseUrl}/{$this->graphApiVersion}/{$pageId}/videos", [
                'access_token' => $accessToken,
                'upload_phase' => 'start',
                'file_size' => $fileSize,
            ]);

            if (!$initResponse->successful()) {
                return $this->handleApiError($initResponse, $account);
            }

            $uploadSessionId = $initResponse->json('upload_session_id');
            $startOffset = 0;
            $endOffset = $initResponse->json('end_offset');

            // Step 2: Upload chunks
            $handle = fopen($filePath, 'rb');
            
            while ($startOffset < $fileSize) {
                $chunkSize = $endOffset - $startOffset;
                fseek($handle, $startOffset);
                $chunk = fread($handle, $chunkSize);

                $chunkResponse = Http::attach(
                    'video_file_chunk',
                    $chunk,
                    'chunk.mp4'
                )->post("{$this->baseUrl}/{$this->graphApiVersion}/{$pageId}/videos", [
                    'access_token' => $accessToken,
                    'upload_phase' => 'transfer',
                    'upload_session_id' => $uploadSessionId,
                    'start_offset' => $startOffset,
                ]);

                if (!$chunkResponse->successful()) {
                    fclose($handle);
                    return $this->handleApiError($chunkResponse, $account);
                }

                $startOffset = $chunkResponse->json('start_offset');
                $endOffset = $chunkResponse->json('end_offset') ?? $fileSize;
            }

            fclose($handle);

            // Step 3: Finish upload
            $finishResponse = Http::post("{$this->baseUrl}/{$this->graphApiVersion}/{$pageId}/videos", [
                'access_token' => $accessToken,
                'upload_phase' => 'finish',
                'upload_session_id' => $uploadSessionId,
                'title' => $title,
                'description' => $description,
            ]);

            if ($finishResponse->successful()) {
                return [
                    'success' => true,
                    'video_id' => $finishResponse->json('id') ?? $finishResponse->json('video_id'),
                    'message' => 'Video uploaded to Facebook successfully',
                ];
            }

            return $this->handleApiError($finishResponse, $account);

        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'exception',
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Handle API error responses
     */
    protected function handleApiError($response, SocialAccount $account): array
    {
        $error = $response->json('error') ?? [];
        $code = $error['code'] ?? 0;
        $message = $error['message'] ?? 'Unknown error';

        // Auth errors
        if (in_array($code, [190, 102, 104])) {
            $account->update(['needs_reauth' => true]);
            return [
                'success' => false,
                'error' => 'auth_error',
                'message' => 'Authentication error: ' . $message,
            ];
        }

        // Rate limit
        if ($code === 32 || $code === 17) {
            return [
                'success' => false,
                'error' => 'quota_exceeded',
                'message' => 'Facebook API rate limit exceeded',
            ];
        }

        return [
            'success' => false,
            'error' => 'api_error',
            'message' => $message,
        ];
    }

    /**
     * Simulate upload for testing
     */
    public function simulateUpload(Video $video): array
    {
        sleep(2);

        if (rand(1, 10) <= 9) {
            return [
                'success' => true,
                'video_id' => 'fb_sim_' . uniqid(),
                'message' => 'Video uploaded to Facebook successfully (simulated)',
            ];
        }

        return [
            'success' => false,
            'error' => 'simulated_error',
            'message' => 'Simulated Facebook upload failure',
        ];
    }
}
