<?php

namespace App\Services;

use App\Models\SocialAccount;
use App\Models\Video;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Exception;

class TikTokUploadService
{
    protected string $baseUrl = 'https://open.tiktokapis.com/v2';

    /**
     * Upload a video to TikTok
     */
    public function upload(Video $video, SocialAccount $account): array
    {
        try {
            $accessToken = decrypt($account->token);

            if ($account->needs_reauth) {
                return [
                    'success' => false,
                    'error' => 'auth_error',
                    'message' => 'Account needs re-authentication',
                ];
            }

            $filePath = storage_path('app/private/' . $video->file_path);
            
            if (!file_exists($filePath)) {
                return [
                    'success' => false,
                    'error' => 'file_not_found',
                    'message' => 'Video file not found',
                ];
            }

            $fileSize = filesize($filePath);
            $title = $video->getTitleForUpload();

            // Step 1: Initialize video upload
            $initResponse = Http::withHeaders([
                'Authorization' => 'Bearer ' . $accessToken,
                'Content-Type' => 'application/json',
            ])->post("{$this->baseUrl}/post/publish/video/init/", [
                'post_info' => [
                    'title' => $title,
                    'privacy_level' => 'PUBLIC_TO_EVERYONE',
                    'disable_duet' => false,
                    'disable_stitch' => false,
                    'disable_comment' => false,
                ],
                'source_info' => [
                    'source' => 'FILE_UPLOAD',
                    'video_size' => $fileSize,
                    'chunk_size' => min($fileSize, 10 * 1024 * 1024), // 10MB max chunk
                    'total_chunk_count' => ceil($fileSize / (10 * 1024 * 1024)),
                ],
            ]);

            if (!$initResponse->successful()) {
                return $this->handleApiError($initResponse, $account);
            }

            $data = $initResponse->json('data');
            $publishId = $data['publish_id'];
            $uploadUrl = $data['upload_url'];

            // Step 2: Upload video file
            $videoContent = file_get_contents($filePath);
            
            $uploadResponse = Http::withHeaders([
                'Content-Type' => 'video/mp4',
                'Content-Range' => "bytes 0-" . ($fileSize - 1) . "/{$fileSize}",
            ])->withBody($videoContent, 'video/mp4')
              ->put($uploadUrl);

            if (!$uploadResponse->successful()) {
                return [
                    'success' => false,
                    'error' => 'upload_failed',
                    'message' => 'Failed to upload video to TikTok',
                ];
            }

            // Step 3: Check publish status
            $statusResponse = Http::withHeaders([
                'Authorization' => 'Bearer ' . $accessToken,
            ])->post("{$this->baseUrl}/post/publish/status/fetch/", [
                'publish_id' => $publishId,
            ]);

            if ($statusResponse->successful()) {
                $status = $statusResponse->json('data.status');
                
                if ($status === 'PUBLISH_COMPLETE') {
                    return [
                        'success' => true,
                        'video_id' => $publishId,
                        'message' => 'Video uploaded to TikTok successfully',
                    ];
                }

                // Still processing - mark as success, TikTok will process async
                return [
                    'success' => true,
                    'video_id' => $publishId,
                    'message' => 'Video submitted to TikTok, processing...',
                ];
            }

            return [
                'success' => true,
                'video_id' => $publishId,
                'message' => 'Video submitted to TikTok',
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'exception',
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Handle API error responses
     */
    protected function handleApiError($response, SocialAccount $account): array
    {
        $error = $response->json('error') ?? [];
        $code = $error['code'] ?? '';
        $message = $error['message'] ?? 'Unknown error';

        // Auth errors
        if (str_contains($code, 'access_token') || $response->status() === 401) {
            $account->update(['needs_reauth' => true]);
            return [
                'success' => false,
                'error' => 'auth_error',
                'message' => 'Authentication error: ' . $message,
            ];
        }

        // Rate limit
        if ($response->status() === 429) {
            return [
                'success' => false,
                'error' => 'quota_exceeded',
                'message' => 'TikTok API rate limit exceeded',
            ];
        }

        return [
            'success' => false,
            'error' => 'api_error',
            'message' => $message,
        ];
    }

    /**
     * Simulate upload for testing
     */
    public function simulateUpload(Video $video): array
    {
        sleep(2);

        if (rand(1, 10) <= 9) {
            return [
                'success' => true,
                'video_id' => 'tt_sim_' . uniqid(),
                'message' => 'Video uploaded to TikTok successfully (simulated)',
            ];
        }

        return [
            'success' => false,
            'error' => 'simulated_error',
            'message' => 'Simulated TikTok upload failure',
        ];
    }
}
