<?php

namespace App\Services;

use App\Models\Schedule;
use App\Models\SocialAccount;
use App\Models\UploadLog;
use App\Models\Video;
use Google\Client;
use Google\Service\YouTube;
use Google\Http\MediaFileUpload;
use Exception;

class YouTubeUploadService
{
    protected ?Client $client = null;

    /**
     * Upload a video to YouTube
     */
    public function upload(Video $video, SocialAccount $account): array
    {
        try {
            // Refresh token if needed
            $this->refreshTokenIfNeeded($account);

            // Check if account needs reauth
            if ($account->needs_reauth) {
                return [
                    'success' => false,
                    'error' => 'auth_error',
                    'message' => 'Account needs re-authentication',
                ];
            }

            // Initialize client
            $client = $this->getClient($account);
            $youtube = new YouTube($client);

            // Prepare video metadata
            $title = $video->getTitleForUpload();
            $description = $video->getDescriptionForUpload();

            // Create video snippet
            $snippet = new YouTube\VideoSnippet();
            $snippet->setTitle($title);
            $snippet->setDescription($description);
            $snippet->setCategoryId('22'); // People & Blogs

            if ($video->tags) {
                $snippet->setTags(explode(',', $video->tags));
            }

            // Set video status (public, private, unlisted)
            $status = new YouTube\VideoStatus();
            $status->setPrivacyStatus('public');
            $status->setSelfDeclaredMadeForKids(false);

            // Create the video resource
            $youtubeVideo = new YouTube\Video();
            $youtubeVideo->setSnippet($snippet);
            $youtubeVideo->setStatus($status);

            // Get video file path
            $filePath = storage_path('app/private/' . $video->file_path);
            
            if (!file_exists($filePath)) {
                return [
                    'success' => false,
                    'error' => 'file_not_found',
                    'message' => 'Video file not found',
                ];
            }

            // Set up chunked upload
            $client->setDefer(true);
            $insertRequest = $youtube->videos->insert('snippet,status', $youtubeVideo);

            $media = new MediaFileUpload(
                $client,
                $insertRequest,
                'video/*',
                null,
                true,
                10 * 1024 * 1024 // 10MB chunks
            );
            $media->setFileSize(filesize($filePath));

            // Upload in chunks
            $status = false;
            $handle = fopen($filePath, 'rb');
            
            while (!$status && !feof($handle)) {
                $chunk = fread($handle, 10 * 1024 * 1024);
                $status = $media->nextChunk($chunk);
            }
            
            fclose($handle);
            $client->setDefer(false);

            if ($status) {
                return [
                    'success' => true,
                    'video_id' => $status->getId(),
                    'message' => 'Video uploaded successfully',
                ];
            }

            return [
                'success' => false,
                'error' => 'upload_failed',
                'message' => 'Upload failed with unknown error',
            ];

        } catch (Exception $e) {
            $errorMessage = $e->getMessage();

            // Check for quota exceeded
            if (str_contains($errorMessage, 'quotaExceeded') || 
                str_contains($errorMessage, 'dailyLimitExceeded') ||
                str_contains($errorMessage, '403')) {
                return [
                    'success' => false,
                    'error' => 'quota_exceeded',
                    'message' => 'YouTube API daily quota exceeded',
                ];
            }

            // Check for auth errors
            if (str_contains($errorMessage, 'invalid_grant') ||
                str_contains($errorMessage, 'Token has been expired') ||
                str_contains($errorMessage, '401')) {
                $account->update(['needs_reauth' => true]);
                return [
                    'success' => false,
                    'error' => 'auth_error',
                    'message' => 'Authentication error. Token may be expired.',
                ];
            }

            return [
                'success' => false,
                'error' => 'exception',
                'message' => $errorMessage,
            ];
        }
    }

    /**
     * Get configured Google client
     */
    protected function getClient(SocialAccount $account): Client
    {
        if ($this->client) {
            return $this->client;
        }

        $client = new Client();
        $client->setClientId(config('services.google.client_id'));
        $client->setClientSecret(config('services.google.client_secret'));
        $client->setAccessToken(decrypt($account->token));

        $this->client = $client;
        return $client;
    }

    /**
     * Refresh token if expired
     */
    protected function refreshTokenIfNeeded(SocialAccount $account): void
    {
        if (!$account->isTokenExpired()) {
            return;
        }

        if (!$account->refresh_token) {
            $account->update(['needs_reauth' => true]);
            return;
        }

        try {
            $client = new Client();
            $client->setClientId(config('services.google.client_id'));
            $client->setClientSecret(config('services.google.client_secret'));
            
            $newToken = $client->fetchAccessTokenWithRefreshToken(decrypt($account->refresh_token));
            
            if (isset($newToken['access_token'])) {
                $account->update([
                    'token' => encrypt($newToken['access_token']),
                    'expires_at' => isset($newToken['expires_in']) 
                        ? now()->addSeconds($newToken['expires_in']) 
                        : null,
                ]);
            }
        } catch (Exception $e) {
            $account->update(['needs_reauth' => true]);
        }
    }

    /**
     * Simulate upload for testing (when no API credentials)
     */
    public function simulateUpload(Video $video): array
    {
        // Simulate processing time
        sleep(2);

        // 90% success rate for simulation
        if (rand(1, 10) <= 9) {
            return [
                'success' => true,
                'video_id' => 'sim_' . uniqid(),
                'message' => 'Video uploaded successfully (simulated)',
            ];
        }

        return [
            'success' => false,
            'error' => 'simulated_error',
            'message' => 'Simulated upload failure for testing',
        ];
    }
}
