<?php

use App\Http\Controllers\DashboardController;
use App\Http\Controllers\InstallController;
use App\Http\Controllers\ScheduleController;
use App\Http\Controllers\SocialAccountController;
use App\Http\Controllers\Admin\AdminUserController;
use App\Http\Controllers\Admin\SettingsController;
use App\Http\Controllers\Admin\SystemController;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\File;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
*/

// Installation Routes (only if not installed)
Route::prefix('install')->name('install.')->group(function () {
    Route::get('/', [InstallController::class, 'welcome'])->name('welcome');
    Route::get('/database', [InstallController::class, 'database'])->name('database');
    Route::post('/database', [InstallController::class, 'saveDatabase'])->name('database.save');
    Route::get('/admin', [InstallController::class, 'admin'])->name('admin');
    Route::post('/admin', [InstallController::class, 'saveAdmin'])->name('admin.save');
    Route::get('/complete', [InstallController::class, 'complete'])->name('complete');
});

// Redirect to installer if not installed
Route::get('/', function () {
    if (!File::exists(storage_path('installed'))) {
        return redirect()->route('install.welcome');
    }
    return view('welcome');
})->name('home');

// Static Pages
Route::get('/about', fn() => view('pages.about'))->name('about');
Route::get('/contact', fn() => view('pages.contact'))->name('contact');
Route::post('/contact', function () {
    // Simple contact form handler - stores in session for now
    return back()->with('success', 'Thank you for your message! We will get back to you soon.');
})->name('contact.send');
Route::get('/blog', fn() => view('pages.blog'))->name('blog');
Route::get('/privacy', fn() => view('pages.privacy'))->name('privacy');
Route::get('/terms', fn() => view('pages.terms'))->name('terms');
Route::get('/cookies', fn() => view('pages.cookies'))->name('cookies');


// Auth routes (using Laravel Breeze/UI)
Route::middleware('guest')->group(function () {
    Route::get('/login', function () {
        if (!File::exists(storage_path('installed'))) {
            return redirect()->route('install.welcome');
        }
        return view('auth.login');
    })->name('login');
    
    Route::post('/login', [\App\Http\Controllers\Auth\LoginController::class, 'login']);
    
    Route::get('/register', function () {
        if (!File::exists(storage_path('installed'))) {
            return redirect()->route('install.welcome');
        }
        return view('auth.register');
    })->name('register');
    
    Route::post('/register', [\App\Http\Controllers\Auth\RegisterController::class, 'register']);
    
    // Password Reset
    Route::get('/forgot-password', function () {
        return view('auth.forgot-password');
    })->name('password.request');
    
    Route::post('/forgot-password', function () {
        // For now, just redirect back with a message
        return back()->with('status', 'If an account exists with that email, you will receive a password reset link.');
    })->name('password.email');
});

Route::post('/logout', function () {
    auth()->logout();
    request()->session()->invalidate();
    request()->session()->regenerateToken();
    return redirect('/');
})->name('logout');

// Authenticated routes
Route::middleware(['auth'])->group(function () {
    
    // Dashboard
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');
    
    // Schedules
    Route::resource('schedules', ScheduleController::class);
    Route::post('schedules/{schedule}/pause', [ScheduleController::class, 'pause'])->name('schedules.pause');
    Route::post('schedules/{schedule}/resume', [ScheduleController::class, 'resume'])->name('schedules.resume');
    
    // Connected Accounts
    Route::get('/accounts', [SocialAccountController::class, 'index'])->name('accounts.index');
    Route::get('/accounts/youtube/redirect', [SocialAccountController::class, 'redirectToYouTube'])->name('accounts.youtube.redirect');
    Route::get('/accounts/youtube/callback', [SocialAccountController::class, 'handleYouTubeCallback'])->name('accounts.youtube.callback');
    Route::get('/accounts/facebook/redirect', [SocialAccountController::class, 'redirectToFacebook'])->name('accounts.facebook.redirect');
    Route::get('/accounts/facebook/callback', [SocialAccountController::class, 'handleFacebookCallback'])->name('accounts.facebook.callback');
    Route::get('/accounts/tiktok/redirect', [SocialAccountController::class, 'redirectToTikTok'])->name('accounts.tiktok.redirect');
    Route::get('/accounts/tiktok/callback', [SocialAccountController::class, 'handleTikTokCallback'])->name('accounts.tiktok.callback');
    Route::delete('/accounts/{account}', [SocialAccountController::class, 'disconnect'])->name('accounts.disconnect');
    
    // User Profile
    Route::get('/profile', [\App\Http\Controllers\ProfileController::class, 'index'])->name('profile.index');
    Route::put('/profile', [\App\Http\Controllers\ProfileController::class, 'update'])->name('profile.update');
    Route::post('/profile/email', [\App\Http\Controllers\ProfileController::class, 'updateEmail'])->name('profile.email');
    Route::post('/profile/password', [\App\Http\Controllers\ProfileController::class, 'updatePassword'])->name('profile.password');
    
    // Subscription / Billing
    Route::get('/subscription', [\App\Http\Controllers\SubscriptionController::class, 'index'])->name('subscription.index');
    Route::post('/subscription/upgrade', [\App\Http\Controllers\SubscriptionController::class, 'upgrade'])->name('subscription.upgrade');
    Route::post('/subscription/cancel', [\App\Http\Controllers\SubscriptionController::class, 'cancel'])->name('subscription.cancel');
    
    // Support Tickets (User)
    Route::get('/support', [\App\Http\Controllers\SupportController::class, 'index'])->name('support.index');
    Route::get('/support/create', [\App\Http\Controllers\SupportController::class, 'create'])->name('support.create');
    Route::post('/support', [\App\Http\Controllers\SupportController::class, 'store'])->name('support.store');
    Route::get('/support/{ticket}', [\App\Http\Controllers\SupportController::class, 'show'])->name('support.show');
    Route::post('/support/{ticket}/reply', [\App\Http\Controllers\SupportController::class, 'reply'])->name('support.reply');
    Route::post('/support/{ticket}/close', [\App\Http\Controllers\SupportController::class, 'close'])->name('support.close');
    
    // Admin routes
    Route::middleware(['admin'])->prefix('admin')->name('admin.')->group(function () {
        
        // User Management
        Route::get('/users', [AdminUserController::class, 'index'])->name('users.index');
        Route::get('/users/{user}', [AdminUserController::class, 'show'])->name('users.show');
        Route::get('/users/{user}/edit', [AdminUserController::class, 'edit'])->name('users.edit');
        Route::put('/users/{user}', [AdminUserController::class, 'update'])->name('users.update');
        Route::delete('/users/{user}', [AdminUserController::class, 'destroy'])->name('users.destroy');
        
        // User Actions
        Route::post('/users/{user}/reset-quota', [AdminUserController::class, 'resetQuota'])->name('users.reset-quota');
        Route::post('/users/{user}/reset-password', [AdminUserController::class, 'resetPassword'])->name('users.reset-password');
        Route::post('/users/{user}/generate-password', [AdminUserController::class, 'generatePassword'])->name('users.generate-password');
        Route::post('/users/{user}/update-email', [AdminUserController::class, 'updateEmail'])->name('users.update-email');
        Route::post('/users/{user}/change-plan', [AdminUserController::class, 'changePlan'])->name('users.change-plan');
        Route::post('/users/{user}/ban', [AdminUserController::class, 'ban'])->name('users.ban');
        Route::post('/users/{user}/unban', [AdminUserController::class, 'unban'])->name('users.unban');
        Route::post('/users/{user}/verify-email', [AdminUserController::class, 'verifyEmail'])->name('users.verify-email');
        Route::post('/users/{user}/unverify-email', [AdminUserController::class, 'unverifyEmail'])->name('users.unverify-email');
        Route::post('/users/{user}/impersonate', [AdminUserController::class, 'impersonate'])->name('users.impersonate');
        Route::post('/users/{user}/clear-data', [AdminUserController::class, 'clearData'])->name('users.clear-data');
        Route::post('/users/{user}/disconnect-accounts', [AdminUserController::class, 'disconnectAccounts'])->name('users.disconnect-accounts');
        Route::get('/users/{user}/export', [AdminUserController::class, 'exportData'])->name('users.export');
        
        // Stop impersonating (accessible from anywhere)
        Route::get('/stop-impersonating', [AdminUserController::class, 'stopImpersonating'])->name('stop-impersonating');
        
        // Plan Management
        Route::get('/plans', [\App\Http\Controllers\Admin\PlanController::class, 'index'])->name('plans.index');
        Route::get('/plans/create', [\App\Http\Controllers\Admin\PlanController::class, 'create'])->name('plans.create');
        Route::post('/plans', [\App\Http\Controllers\Admin\PlanController::class, 'store'])->name('plans.store');
        Route::get('/plans/{plan}/edit', [\App\Http\Controllers\Admin\PlanController::class, 'edit'])->name('plans.edit');
        Route::put('/plans/{plan}', [\App\Http\Controllers\Admin\PlanController::class, 'update'])->name('plans.update');
        Route::delete('/plans/{plan}', [\App\Http\Controllers\Admin\PlanController::class, 'destroy'])->name('plans.destroy');
        
        // Subscriptions & Revenue
        Route::get('/subscriptions', [\App\Http\Controllers\Admin\SubscriptionController::class, 'index'])->name('subscriptions.index');
        Route::post('/payments/{payment}/approve', [\App\Http\Controllers\Admin\SubscriptionController::class, 'approve'])->name('payments.approve');
        Route::post('/payments/{payment}/reject', [\App\Http\Controllers\Admin\SubscriptionController::class, 'reject'])->name('payments.reject');
        
        // CMS Settings
        Route::get('/settings', [SettingsController::class, 'index'])->name('settings.index');
        Route::put('/settings', [SettingsController::class, 'update'])->name('settings.update');
        
        // System
        Route::get('/system/failed-jobs', [SystemController::class, 'failedJobs'])->name('system.failed-jobs');
        Route::post('/system/failed-jobs/{id}/retry', [SystemController::class, 'retryJob'])->name('system.retry-job');
        Route::delete('/system/failed-jobs/{id}', [SystemController::class, 'deleteJob'])->name('system.delete-job');
        Route::get('/system/logs', [SystemController::class, 'logs'])->name('system.logs');
        
        // Setup Guide
        Route::get('/setup-guide', fn() => view('admin.setup-guide'))->name('setup-guide');
        
        // API Keys Management
        Route::get('/api-keys', [\App\Http\Controllers\Admin\ApiKeyController::class, 'index'])->name('api-keys.index');
        Route::post('/api-keys/social', [\App\Http\Controllers\Admin\ApiKeyController::class, 'updateSocialKeys'])->name('api-keys.social.update');
        
        // Payment Management
        Route::get('/payments', [\App\Http\Controllers\Admin\PaymentController::class, 'payments'])->name('payments.index');
        Route::get('/payments/{payment}', [\App\Http\Controllers\Admin\PaymentController::class, 'show'])->name('payments.show');
        Route::post('/payments/{payment}/approve', [\App\Http\Controllers\Admin\PaymentController::class, 'approve'])->name('payments.approve');
        Route::post('/payments/{payment}/reject', [\App\Http\Controllers\Admin\PaymentController::class, 'reject'])->name('payments.reject');
        Route::get('/subscriptions', [\App\Http\Controllers\Admin\PaymentController::class, 'subscriptions'])->name('subscriptions.index');
        Route::get('/gateways', [\App\Http\Controllers\Admin\PaymentController::class, 'gateways'])->name('payments.gateways');
        Route::put('/gateways/{gateway}', [\App\Http\Controllers\Admin\PaymentController::class, 'updateGateway'])->name('payments.gateways.update');
        
        // Batch Management
        Route::get('/batches', [\App\Http\Controllers\Admin\BatchController::class, 'index'])->name('batches.index');
        Route::post('/batches/{schedule}/stop', [\App\Http\Controllers\Admin\BatchController::class, 'stop'])->name('batches.stop');
        Route::post('/batches/{schedule}/resume', [\App\Http\Controllers\Admin\BatchController::class, 'resume'])->name('batches.resume');
        Route::post('/batches/stop-all', [\App\Http\Controllers\Admin\BatchController::class, 'stopAll'])->name('batches.stop-all');
        Route::post('/batches/test-cron', [\App\Http\Controllers\Admin\BatchController::class, 'testCron'])->name('batches.test-cron');
        
        // Ad Management
        Route::get('/ads', [\App\Http\Controllers\Admin\AdController::class, 'index'])->name('ads.index');
        Route::put('/ads/{ad}', [\App\Http\Controllers\Admin\AdController::class, 'update'])->name('ads.update');
        
        // Support Tickets (Admin)
        Route::get('/support', [\App\Http\Controllers\Admin\SupportController::class, 'index'])->name('support.index');
        Route::get('/support/{ticket}', [\App\Http\Controllers\Admin\SupportController::class, 'show'])->name('support.show');
        Route::post('/support/{ticket}/reply', [\App\Http\Controllers\Admin\SupportController::class, 'reply'])->name('support.reply');
        Route::post('/support/{ticket}/status', [\App\Http\Controllers\Admin\SupportController::class, 'updateStatus'])->name('support.status');
        Route::post('/support/{ticket}/assign', [\App\Http\Controllers\Admin\SupportController::class, 'assign'])->name('support.assign');
    });
});
